# -*- coding: utf-8 -*-
import datetime
import json
import logging
import os

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.hollywood.MakeHollywoodAmmo import MakeHollywoodAmmo
from sandbox.projects.websearch.begemot import resources as begemot_resources
from sandbox.projects.websearch.upper import resources as upper_resources


logger = logging.getLogger(__name__)


class MakeBegemotAmmo(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Task for making Begemot ammo (grepping apphost service requests from YT) using search/tools/request_sampler.
    """

    class Requirements(sdk2.Task.Requirements):
        ram = 10 * 1024  # 10 Gb
        disk_space = 30 * 1024  # 10 Gb

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60 * 60
        request_sampler = sdk2.parameters.Resource(
            "request_sampler executable. Leave it empty to use the latest released:stable.",
            resource_type=upper_resources.REQUEST_SAMPLER_EXECUTABLE,
            required=False,
        )
        input_table = sdk2.parameters.String(
            "Input yt table.",
            description="(By default yesterday 18) -i flag of request_sampler. For example (a Hollywood log on hahn):\n"
            "//logs/apphost-alice-rtlog/1h/2022-06-08T18:00:00",
            required=False,
        )
        number_of_requests = sdk2.parameters.Integer(
            "Number of requests (of each kind) to get.",
            description="Both -d and -g flags of request_sampler. Value 0 - all.",
            required=False,
            default=30000,
        )
        yt_token_vault_key = sdk2.parameters.String(
            "Vault secret with YT token.",
            description="This must be a Sandbox Valt secret.",
            default="robot-voiceint_yt_token",
        )
        remote_temp_tables_directory = sdk2.parameters.String(
            "Directory for temp tables while running request sampler",
            required=False
        )
        remote_temp_files_directory = sdk2.parameters.String(
            "Directory for temp files while running request sampler",
            required=False
        )
        ext_params = binary_task.LastBinaryReleaseParameters(stable=True)

    def on_save(self):
        super(MakeBegemotAmmo, self).on_save()

    def on_execute(self):
        super(MakeBegemotAmmo, self).on_execute()

        with self.memoize_stage.ammo_subtask():
            subtask = MakeHollywoodAmmo(
                self,
                owner=self.Parameters.owner,
                request_sampler=self.Parameters.request_sampler,
                input_table=self.get_input_table(),
                number_of_requests=self.Parameters.number_of_requests,
                graphs_sources=self.get_begemot_graphs_sources(),
                frame_base_64_encoded=True,
                frame_column_name='event_binary',
                sources_types='regular',
                yt_token_vault_key=self.Parameters.yt_token_vault_key,
                process_lines='merge',
                shuffle_lines=True,
                output_format='app-host-json',
                converter_input_format='service_request',
                binary_executor_release_type='stable',
                remote_temp_tables_directory=self.Parameters.remote_temp_tables_directory,
                remote_temp_files_directory=self.Parameters.remote_temp_files_directory,
            ).enqueue()
            self.Context.subtask_id = subtask.id
            if 1 + 1 == 2:  # To be able to see syntax highlighting beneath
                raise sdk2.WaitTask(subtask, list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), wait_all=True, timeout=60 * 60 * 3)

        with self.memoize_stage.convert_to_begemot_format():
            subtask = self.find(id=self.Context.subtask_id).first()
            subtask_ammo_path = sdk2.ResourceData(sdk2.Resource["HOLLYWOOD_AMMO"].find(task=subtask).first()).path
            with open(os.path.join(str(subtask_ammo_path), 'ammo.out'), 'r') as f:
                converted_lines = [json.dumps(json.loads(line)["answers"]) for line in f.readlines()]
            logger.info("MAKE_HOLLYWOOD_AMMO grepped {} lines".format(len(converted_lines)))
            output_file = 'begemot_megamind_yp_production_sas_apphost_{}.txt'.format(len(converted_lines))
            begemot_resources.BEGEMOT_APPHOST_QUERIES(
                self,
                "Apphost requests for shard Megamind collected by MAKE_BEGEMOT_AMMO task from apphost eventlogs",
                output_file,
            )
            with open(output_file, 'w') as f:
                for line in converted_lines:
                    f.write("{}\n".format(line))

    def get_input_table(self):
        if self.Parameters.input_table:
            return self.Parameters.input_table
        t = datetime.datetime.now() - datetime.timedelta(hours=24, minutes=0)
        t = t.replace(hour=18, minute=0, second=0, microsecond=0)
        return "//logs/apphost-alice-rtlog/1h/{}".format(t.isoformat())

    def get_begemot_graphs_sources(self):
        # TODO(@yagafarov) do not hardcode (use for example mit library)
        return 'megamind_begemot_stage:BEGEMOT_WORKER_MEGAMIND.BEGEMOT_WORKER_MEGAMIND_MERGER'
