# -*- coding: utf-8 -*-

import datetime
import time
import logging

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

MAX_TAGS = 10


def get_tags_to_save(releases_meta):
    important_tags = []
    tag_names = [
        'current_tag',
        'prev_tag'
    ]
    for tag_name in tag_names:
        if tag_name in releases_meta:
            important_tags.append(releases_meta[tag_name])
    return important_tags


def releases_meta_cleanup(yt_client, releases_meta_path):
    releases_meta = yt_client.get(releases_meta_path)
    releases = releases_meta['releases']

    # sort tags in descending order, new ones in the beginning, old ones in the end
    delete_candidates = sorted(releases.keys(), reverse=True)
    delete_list = delete_candidates[MAX_TAGS:]

    tags_to_save = get_tags_to_save(releases_meta)

    releases_path = releases_meta_path + '/releases'
    for tag_to_delete in delete_list:
        if tag_to_delete not in tags_to_save:
            logging.info('Removing tag {}'.format(tag_to_delete))
            yt_client.remove(releases_path + '/' + tag_to_delete)


def write_release_to_yt_attrs(tag, resources, yt_server, yt_prefix, vault_yt_token=None):
    import yt.wrapper as yt
    if vault_yt_token is not None:
        yt_client = yt.YtClient(proxy=yt_server, token=sdk2.Vault.data(vault_yt_token))
    else:
        yt_client = yt.YtClient(proxy=yt_server)

    canonized_tag = tag.replace('/', '-')
    releases_meta_path = yt_prefix + '/@releases_meta'
    releases_path = releases_meta_path + '/releases'
    current_release_tag_path = releases_meta_path + '/current_tag'
    prev_release_tag_path = releases_meta_path + '/prev_tag'
    current_release_path = releases_path + '/' + canonized_tag

    if not yt_client.exists(releases_meta_path):
        logging.info('Creating releases meta root at {}'.format(releases_meta_path))
        yt_client.set(releases_meta_path, {})

    if not yt_client.exists(releases_path):
        logging.info('Creating releases info at {}'.format(releases_path))
        yt_client.set(releases_path, {})

    if yt_client.exists(current_release_tag_path):
        curr_tag = yt_client.get(current_release_tag_path)
        if curr_tag == canonized_tag:
            logging.info(
                'Trying to release tag {}, but it is already the current release tag. Will do nothing'.format(tag)
            )
            return

    now = datetime.datetime.now()
    now_str = now.strftime('%Y-%m-%d %H:%M:%S')
    now_timestamp = int(time.mktime(now.timetuple()))

    release_info = {
        'tag': tag,
        'timestr': now_str,
        'timestamp': now_timestamp,
        'resources': resources,
    }
    logging.info('Adding release info for tag {}'.format(tag))
    yt_client.set(current_release_path, release_info)

    if yt_client.exists(current_release_tag_path):
        yt_client.set(prev_release_tag_path, yt_client.get(current_release_tag_path))

    yt_client.set(current_release_tag_path, canonized_tag)
    logging.info('current_tag is {}'.format(yt_client.get(current_release_tag_path)))
    if yt_client.exists(prev_release_tag_path):
        logging.info("prev_tag is {}".format(yt_client.get(prev_release_tag_path)))

    releases_meta_cleanup(yt_client, releases_meta_path)


class AltaySetReleaseMeta(sdk2.Task):
    '''
    Add some attributes to released resources to make altay release cycle more convinient.
    '''
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # Mb
        environments = [
            PipEnvironment("yandex-yt", use_wheel=True),
        ]

    class Parameters(sdk2.Task.Parameters):
        '''
        Task parameters:
            :param component_resources: Dictionary of resources in the following format { resource_name: res_id, ...}
            :param yt_prefix: Altay db YT prefix
            :param yt_server: Name of YT cluster
            :param tag: Branch tag for the released resources
            :param vault_yt_token: YT token item name in secure vault
        '''
        component_resources = sdk2.parameters.Dict(label="Resources", required=True)

        yt_prefix = sdk2.parameters.String(label="YT prefix", default="//home/altay/db")
        yt_server = sdk2.parameters.String(label="YT proxy server", default="hahn.yt.yandex.net")
        tag = sdk2.parameters.String(
            label="Tag",
            required=True,
            description="Branch tag in format altay/stable-<branch>-<tag>",
        )
        vault_yt_token = sdk2.parameters.String(
            label="YT token vault item name",
            required=True,
            description="Name of Sandbox vault item with YT token allowing write to yt_prefix path on yt_server",
        )

    def on_execute(self):
        '''
        Write meta information to attributes on YT according to given prefix and list of released resources
        '''

        # Prepare the release meta record
        write_release_to_yt_attrs(
            self.Parameters.tag,
            self.Parameters.component_resources,
            self.Parameters.yt_server,
            self.Parameters.yt_prefix,
            vault_yt_token=self.Parameters.vault_yt_token
        )
