import os
import re
from shutil import copyfile

from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE
from sandbox.sandboxsdk.process import run_process
from sandbox import sdk2

from sandbox.projects.answers.AnswersFrontendBase import AnswersFrontendBase, resources
from sandbox.projects.answers.AnswersFrontendKarma import AnswersFrontendKarma
from sandbox.projects.answers.AnswersFrontendMocha import AnswersFrontendMocha
from sandbox.projects.answers.AnswersFrontendTslint import AnswersFrontendTslint
from sandbox.projects.answers.AnswersFrontendStylelint import AnswersFrontendStylelint
from sandbox.projects.answers.AnswersFrontendTypescript import AnswersFrontendTypescript
from sandbox.projects.answers.AnswersFrontendJest import AnswersFrontendJest
from sandbox.projects.answers.AnswersFrontendE2E import AnswersFrontendE2E
from sandbox.projects.answers.AnswersFrontendGemini import AnswersFrontendGemini
from sandbox.projects.answers.AnswersFrontendPalmsync import AnswersFrontendPalmsync

from sandbox.projects.collections.CollectionsFrontend import CollectionsFrontend


YASTATIC_ANSWERS_URL = {
    'production': 'https://yastatic.net/answers/_/',
    'testing': 'https://betastatic.yastatic.net/answers/_/',
}


class AnswersFrontend(AnswersFrontendBase, CollectionsFrontend):
    """
        Answers frontend build
    """
    class Parameters(CollectionsFrontend.Parameters):
        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_14.04_trusty',  # necessary for correct default resource searching
            required=True,
        )

        with sdk2.parameters.Group('Hermione') as hermione:
            hermione_platforms = sdk2.parameters.List(
                'Hermione platforms', required=True, default=('mobile', 'desktop'),
            )

    def _build(self, project_env, environment, resource, **kwargs):
        super(AnswersFrontend, self)._build(project_env, environment, resource, **kwargs)

        env = self.Parameters.build_env or {}
        env.update(environment)
        env['GITHUB_PR_NUMBER'] = self.pr_number

        self._calc_hermione_coverage(
            project_env=project_env,
            environment=env,
        )

    def _build_tslint(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendTslint)
        )

    def _build_mocha(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendMocha, platform='mobile')
        )
        self.enqueue_task(
            self.create_task(AnswersFrontendMocha, platform='desktop')
        )

    def _build_karma(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendKarma, platform='mobile')
        )
        self.enqueue_task(
            self.create_task(AnswersFrontendKarma, platform='desktop')
        )

    def _build_tsc(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendTypescript)
        )

    def _build_gemini(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendGemini, platform='mobile')
        )
        self.enqueue_task(
            self.create_task(AnswersFrontendGemini, platform='desktop')
        )

    def _build_e2e_components(self):
        for platform in self.Parameters.hermione_platforms:
            self.enqueue_task(
                self.create_task(AnswersFrontendE2E, preset=platform, report='hermione-report')
            )

    def _build_e2e_features(self):
        for platform in self.Parameters.hermione_platforms:
            self.enqueue_task(
                self.create_task(
                    AnswersFrontendE2E,
                    preset='features:{}'.format(platform),
                    semaphore_name='answers_features_e2e',
                    report='hermione-report'
                )
            )

    def _build_palmsync(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendPalmsync, app_ver=self.Context.app_ver, git_lfs=True)
        )

    def _build_eslint(self):
        pass

    def _build_stylelint(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendStylelint)
        )

    def _build_jest(self):
        self.enqueue_task(
            self.create_task(AnswersFrontendJest, platform='mobile')
        )
        self.enqueue_task(
            self.create_task(AnswersFrontendJest, platform='desktop')
        )

    def _build_chrome(self):
        pass

    def _build_deploy(self):
        pass

    def _calc_hermione_coverage(self, environment, project_env):
        target = self.pr_number

        if not target:
            target = 'master'

        run_process(
            [self.node_bin, self.gulp_bin, 'hermione:coverage' if target == 'master' else 'hermione:coverage:github', '--' + project_env],
            log_prefix='hermione_coverage',
            work_dir=str(self.src_root),
            environment=environment
        )

        if target == 'master':
            coverage = sdk2.ResourceData(resources.AnswersFrontendHermioneCoverage(
                self,
                '[{}] Hermione coverage'.format(project_env),
                'coverage-{}.json'.format(project_env),
                env=project_env,
            ))

            if self.Parameters.use_ram_drive:
                source_path = str(self.working_path(self.project_dir, 'dist/web/hermione-coverage.json'))
            else:
                source_path = os.path.join(self.src_root, 'dist/web/hermione-coverage.json')

            copyfile(
                str(source_path),
                str(coverage.path)
            )

            coverage.ready()

    def _build_yastatic_s3(self, project_env):
        pass

    def _release_yastatic_s3(self, release_status):
        pass

    def _build_yastatic(self, project_env):
        self._build_yastatic_project(
            project_env=project_env,
            prj='answers'
        )

    def yastatic_meta(self, project_env, prj='pdb'):
        base_dir = os.path.join('~/', project_env)

        version = re.findall(r'^v(.+?)$', self.Context.tag)[0]

        if project_env != 'production':
            version = version + '-' + project_env

        return {
            'package_name': 'yandex-static-answers',
            'current_version': version,
            'build_dir': os.path.join(base_dir, 'build/answers'),

            'src_root_debian_dir': os.path.join(self.src_root, 'debian-answers'),
            'src_root_static_dir': os.path.join(self.src_root, 'dist/web/client/znatoki'),

            'yastatic_url': YASTATIC_ANSWERS_URL[project_env]
        }
