# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
from sandbox import common
from sandbox.common.types import task as ctt
import sandbox.common.types.misc as ctm
import shutil

from sandbox.projects.sandbox_ci import SANDBOX_CI_ARTIFACT
from sandbox.projects.answers.AnswersFrontendBase import AnswersFrontendBase
from sandbox.projects.collections.CollectionsFrontendBase.base import CollectionsFrontendBase
from sandbox.projects.collections.CollectionsFrontendBase.resources import CollectionsFrontendResourceE2EReport
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process
from sandbox.common import config
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE
from sandbox.common.errors import TaskFailure

DEFAULT_TUNNELER_HOSTS = '["tun.si.yandex-team.ru"]'
FALLBACK_REPORT_NAME = 'gemini-report'


class AnswersFrontendE2E(AnswersFrontendBase, CollectionsFrontendBase):
    """
        Answers E2E hermione tests
    """

    name = 'ANSWERS_FRONTEND_E2E'

    class Requirements(CollectionsFrontendBase.Requirements):
        dns = ctm.DnsType.LOCAL
        ram = 12 * 1024
        cores = 4

    class Parameters(CollectionsFrontendBase.Parameters):
        kill_timeout = 2 * 60 * 60
        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_16.04_xenial',  # necessary for correct default resource searching
            required=True,
        )

        with CollectionsFrontendBase.Parameters.github() as github:
            git_lfs = sdk2.parameters.Bool('Git Lfs', default=True)

        with sdk2.parameters.Group('Features') as features:
            preset = sdk2.parameters.String('Preset', default='mobile')
            report = sdk2.parameters.String('Report', default='e2e-report')
            semaphore_name = sdk2.parameters.String('Semaphore name', default=None)
            flags = sdk2.parameters.List('Flags', default=[])
            custom_opts = sdk2.parameters.String('Command options (flags analog for castle)')

        with sdk2.parameters.Group('Repeater') as repeater:
            repeat = sdk2.parameters.Integer('Repeat count', default=0)
            grep = sdk2.parameters.String('Grep', default='')

        with sdk2.parameters.Group('Tunneler settings') as tunneler_settings:
            tunneler_user = sdk2.parameters.String('Tunneller user', default='robot-serp-bot')
            tunneler_hosts = sdk2.parameters.String('Tunneller hosts', default=DEFAULT_TUNNELER_HOSTS)

    def get_secret(self, name):
        return sdk2.Vault.data('YASAP', 'e2e_' + name)

    def get_report_http_proxy_link(self, report):
        return common.rest.Client().resource[report.id].read()['http']['proxy']

    class Context(CollectionsFrontendBase.Context):
        report_url = ''

    def create_report_resource(self):
        report_path = os.path.join(self.src_root, self.Parameters.report)

        if not os.path.isdir(report_path):
            report_path = os.path.join(self.src_root, FALLBACK_REPORT_NAME)
            if not os.path.isdir(report_path):
                raise TaskFailure('Could not find report dir')

        try:
            shutil.copytree(report_path, os.path.join(self.src_root, 'e2e-report-json'))
            shutil.copytree(report_path, os.path.join(self.src_root, 'e2e-report-html'))

            res = CollectionsFrontendResourceE2EReport(
                self,
                'E2E report: {}'.format(self.Parameters.preset),
                report_path,
                preset=self.Parameters.preset,
                commit=self.Parameters.hash,
            )

            report = sdk2.ResourceData(res)
            report.ready()

            report_url = '{}/index.html'.format(self.get_report_http_proxy_link(res)) if report else ''

            if report_url:
                self.Context.report_url = report_url

            hermione_json_path = os.path.join(self.src_root, 'e2e-report-json/report.json')

            if os.path.isfile(hermione_json_path):
                hermione_json_res = SANDBOX_CI_ARTIFACT(
                    task=self,
                    description='E2E report: {} json'.format(self.Parameters.preset),
                    path=hermione_json_path,
                    type='json-reporter',
                    tool='hermione',
                    project='collections',
                    branch=self.Context.feature_branch,
                )

                json_report = sdk2.ResourceData(hermione_json_res)
                json_report.ready()

                hermione_res = SANDBOX_CI_ARTIFACT(
                    task=self,
                    description='E2E report: {} hermione'.format(self.Parameters.preset),
                    path=os.path.join(self.src_root, 'e2e-report-html'),
                    type='hermione-report',
                )

                hermione_report = sdk2.ResourceData(hermione_res)
                hermione_report.ready()

        except Exception as e:
            raise TaskFailure('Could not make resource {}'.format(e))

    def get_testcop_project(self):
        return None

    def get_issue_keys(self, environment):
        return None

    @sdk2.header()
    def header(self):
        link_template = '<h4><a href="{url}" target="_blank">{text}</a></h4>'
        report_url = self.Context.report_url
        preset = self.Parameters.preset

        if report_url:
            report_link = link_template.format(url=report_url, text='E2E Report: ' + str(preset))

            testcop_project = self.get_testcop_project()
            testcop_link = link_template.format(
                url='https://testcop.si.yandex-team.ru/task/{}?project={}'.format(self.id, testcop_project),
                text='Testcop'
            ) if testcop_project else ''

            return report_link + testcop_link
        else:
            return ''

    def on_enqueue(self):
        super(AnswersFrontendE2E, self).on_enqueue()

        if self.Parameters.semaphore_name:
            with self.memoize_stage.semaphore:
                self.Requirements.semaphores = ctt.Semaphores(
                    acquires=[
                        ctt.Semaphores.Acquire(name=self.Parameters.semaphore_name, weight=1),
                    ],
                    release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH, ctt.Status.Group.WAIT, ctt.Status.FINISHING),
                )

    def on_execute(self):
        preset = self.Parameters.preset
        context = '[Collections CI] E2E: {}'.format(preset)

        self.report_pending_to_github(context)

        try:
            self.prepare_working_copy()
            self.install_modules()

            environment = self.environment
            environment['TUNNELER_USERNAME'] = self.Parameters.tunneler_user
            environment['TUNNELER_HOST_ID'] = config.Registry().this.id
            environment['tunneler_hosts'] = self.Parameters.tunneler_hosts
            environment['tunneler_version'] = '2'
            environment['tunneler_cloud'] = 'yp'
            environment['SANDBOX_AUTH_TOKEN'] = self.get_secret('sandbox_auth_token')
            environment['TUS_OAUTH_TOKEN'] = self.get_secret('tus_oauth_token')
            environment['STARTREK_TOKEN'] = sdk2.Vault.data('YASAP', 'startrek_token')
            environment['SANDBOX_E2E_PRESET'] = str(self.Parameters.preset)
            environment['ADMIN_BACKEND_OAUTH_TOKEN'] = sdk2.Vault.data('YASAP', 'admin_backend_oauth_token')

            issue_key = self.get_issue_keys(environment)

            if issue_key:
                environment['ISSUE_KEYS'] = issue_key

            repeat_count = int(self.Parameters.repeat)

            if repeat_count > 0:
                environment['hermione_test_repeater_enabled'] = 'true'

            with sdk2.ssh.Key(self, 'SANDBOX_CI_SEARCH_INTERFACES', 'robot-serp-bot.id_rsa'):
                run_params = [
                    self.node_bin,
                    self.gulp_bin,
                    'e2e:ci',
                    '--set',
                    preset
                ]

                if self.Parameters.flags:
                    run_params = run_params + self.Parameters.flags

                if repeat_count > 0:
                    run_params = run_params + [
                        '--retry', '0',
                        '--grep', self.Parameters.grep,
                        '--repeat', str(self.Parameters.repeat)
                    ]

                if self.Parameters.custom_opts:
                    run_params = run_params + self.Parameters.custom_opts.split(' ')

                run_process(
                    run_params,
                    work_dir=self.src_root,
                    log_prefix='e2e',
                    environment=environment
                )
                self.create_report_resource()

        except SandboxSubprocessError:
            self.report_error_to_github(context)
            self.create_report_resource()
            raise

        self.report_success_to_github(context)


__TASK__ = AnswersFrontendE2E
