# -*- coding: utf-8 -*-

from sandbox import common
import sandbox.common.types.misc as ctm
import datetime
import logging
import os
from sandbox import sdk2
import shutil
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.answers import resources
from sandbox.projects.answers.common.encrypt_mixin import EncryptMixin, GpgSettings


class AnswersPsqlDump(sdk2.Task, EncryptMixin):
    """Create Answers (Znatoki) dump of PostgreSQL"""

    class Parameters(sdk2.Parameters):
        dbname = parameters.String('Database Name', required=True)
        host = parameters.String('Database Host', required=True)
        is_series = parameters.Bool('Is series database?', default=False)
        port = parameters.Integer(
            'Database Port', required=True, default=12000
        )
        username = parameters.String('Database User', required=True)
        password = parameters.String(
            'Database User Password Secret', required=True
        )
        postgres_resource = parameters.Resource(
            'Resource with PSQL',
            resource_type=resources.AnswersPostgresql,
            required=True,
        )
        ramdrive_size = parameters.Integer('RamDrive size in GB', default=4)
        gpg_key_owner = parameters.String(
            'Gpg Key Owner',
            required=True,
        )
        env = parameters.String(
            'Database environment',
            choices=[
                ('dev', resources.Environments.DEV),
                ('prod', resources.Environments.PROD),
                ('prestable', resources.Environments.PRESTABLE),
            ],
            required=True,
        )

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        if self.Parameters.ramdrive_size:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                int(self.Parameters.ramdrive_size) << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))

    def dump_database(self, psql_path):
        pg_dump_path = os.path.join(psql_path, 'bin/pg_dump')
        dump_path = os.path.join(os.getcwd(), '{}_dump'.format(self.Parameters.dbname))
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('psql_dump')
        ) as pl:
            psql_password = sdk2.Vault.data(self.Parameters.password)
            dump = subprocess.Popen(
                [
                    pg_dump_path,
                    '--dbname={}'.format(self.Parameters.dbname),
                    '--host={}'.format(self.Parameters.host),
                    '--port={}'.format(self.Parameters.port),
                    '--username={}'.format(self.Parameters.username),
                    '--password',
                    '--format=d',
                    '--schema=public',
                    '--exclude-table={}'.format('repl_mon'),
                    '--exclude-table=_*',
                    '--file={}'.format(dump_path)

                ],
                stdin=subprocess.PIPE,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            dump.communicate(psql_password)
            exitcode = dump.wait()
        if exitcode:
            raise Exception('Dump failed with exitcode %s' % exitcode)
        if not os.path.exists(dump_path):
            raise Exception('Empty dump %s' % dump_path)
        return dump_path

    def on_execute(self):
        task_path = os.getcwd()
        settings = GpgSettings(
            key_owner='YASAP',
            secret_key_name='answers_pgp_private_key',
            public_key_name='answers_pgp_public_key',
            recipient=self.Parameters.gpg_key_owner,
        )
        local_psql_path = str(
            sdk2.ResourceData(
                sdk2.Resource[self.Parameters.postgres_resource]
            ).path
        )
        self.setup_ramdrive()
        dump_path = self.dump_database(local_psql_path)
        encrypted_dump = self.encrypt(dump_path, settings)

        dump_ramdrive_path = os.path.join(os.getcwd(), encrypted_dump)
        os.chdir(task_path)
        if dump_ramdrive_path != os.path.join(os.getcwd(), encrypted_dump):
            shutil.copyfile(dump_ramdrive_path, encrypted_dump)

        if self.Parameters.is_series:
            resource = resources.AnswersSeriesPostgresqlDump
        else:
            resource = resources.AnswersPostgresqlDump

        resource_metadata = resource(
            self,
            "Answers (Znatoki) PSQL dump by {}".format(
                datetime.datetime.utcnow().isoformat()
            ),
            encrypted_dump,
            env=self.Parameters.env,
        )
        resource_data = sdk2.ResourceData(resource_metadata)
        resource_data.ready()
