# -*- coding: utf-8 -*-
from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import logging
import os
from sandbox import sdk2
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.answers import resources
from sandbox.projects.answers.common.encrypt_mixin import EncryptMixin, GpgSettings


class AnswersRestoreDB(sdk2.Task, EncryptMixin):
    """Restores Answers (Znatoki) dump of PostgreSQL to db"""

    class Parameters(sdk2.Parameters):
        kill_timeout = 12 * 3600  # 12 hours
        dbname = parameters.String('Load Database Name', required=True)
        host = parameters.String('Load Database Host', required=True)
        port = parameters.Integer(
            'Load Database Port', required=True, default=12000
        )
        is_series = parameters.Bool('Is series database', default=False)
        username = parameters.String('Load Database User', required=True)
        password = parameters.String(
            'Load Database User Password Secret', required=True
        )
        postgres_resource = parameters.Resource(
            'Resource with PSQL',
            resource_type=resources.AnswersPostgresql,
            required=True,
        )
        postgres_dump_resource = parameters.Resource(
            'PG dump resource',
            resource_type=resources.AnswersSeriesPostgresqlDump if is_series else resources.AnswersPostgresqlDump,
            state=(ctr.State.READY,),
            required=False,
            description='The dump you want to restore. The most recent is taken by default'
        )
        ramdrive_size = parameters.Integer('RamDrive size in GB', default=4)
        gpg_key_owner = parameters.String(
            'Gpg Key Owner',
            required=True,
        )
        env = parameters.String(
            'Restored Database environment',
            choices=[
                ('dev', resources.Environments.DEV),
                ('prod', resources.Environments.PROD),
                ('prestable', resources.Environments.PRESTABLE),
            ],
            required=True,
        )

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        if self.Parameters.ramdrive_size:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                int(self.Parameters.ramdrive_size) << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: %s path: %s',
                common.utils.size2str(self.ramdrive.size << 20),
                self.ramdrive.path,
            )
            os.chdir(str(self.ramdrive.path))

    def get_dump_path(self):
        if self.Parameters.postgres_dump_resource:
            dump_resource = self.Parameters.postgres_dump_resource
        else:
            if self.Parameters.is_series:
                dump_resource = sdk2.Resource.find(
                    resources.AnswersSeriesPostgresqlDump,
                    attrs={'env': self.Parameters.env},
                    state=ctr.State.READY,
                ).order(
                    -resources.AnswersSeriesPostgresqlDump.id,
                ).first()
            else:
                dump_resource = sdk2.Resource.find(
                    resources.AnswersPostgresqlDump,
                    attrs={'env': self.Parameters.env},
                    state=ctr.State.READY,
                ).order(
                    -resources.AnswersPostgresqlDump.id,
                ).first()
        result = str(sdk2.ResourceData(dump_resource).path)
        logging.info('Using dump resource: %s', result)
        settings = GpgSettings(
            key_owner='YASAP',
            secret_key_name='answers_pgp_private_key',
            public_key_name='answers_pgp_public_key',
            recipient=self.Parameters.gpg_key_owner,
        )
        return self.decrypt(result, settings)

    def _find_host_restore(self, hosts, port, dbname, username, password, psql_path, pl):
        splited_hosts = hosts.split(',')
        master_host = splited_hosts[0]
        for host in splited_hosts:
            fetch = subprocess.Popen(
                [
                    psql_path,
                    '--dbname={}'.format(dbname),
                    '--host={}'.format(host),
                    '--port={}'.format(port),
                    '--username={}'.format(username),
                    '-t',
                    '-c',
                    'select pg_is_in_recovery()'
                ],
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
            )

            rows = fetch.communicate(password)[0].split('\n')[0]
            exitcode = fetch.wait()
            if exitcode:
                raise Exception(
                    'Failed to fetch database table, exitcode: {}'.format(
                        exitcode
                    )
                )

            logging.info("Query response: {}".format(rows))
            if rows == ' f':
                master_host = host
                break
        return master_host

    def run_restore_with_password(self, args, password, log_pipe):
        process = subprocess.Popen(
            args, stdin=subprocess.PIPE, stdout=log_pipe.stdout, stderr=log_pipe.stderr,
        )
        process.communicate(password)
        exitcode = process.wait()
        if exitcode:
            raise Exception('Failed to restore database, exitcode: {}'.format(exitcode))
        logging.info("{} was ended with exit code {}".format(args[0], exitcode))

    def run_pg_command_with_password(
        self, command, host, port, dbname, username, password, args, log_pipe
    ):
        command_args = [command]
        command_args.append(
            'host={host} '
            'port={port} '
            'dbname={database} '
            'user={user} '
            'target_session_attrs=read-write'.format(
                host=host,
                port=port,
                database=dbname,
                user=username
            )
        )
        command_args.extend(args)

        env = dict(os.environ)
        env['PGPASSWORD'] = password

        process = subprocess.Popen(
            command_args,
            stdin=subprocess.PIPE,
            stdout=log_pipe.stdout,
            stderr=log_pipe.stderr,
            env=env,
        )
        exitcode = process.wait()
        if exitcode:
            raise Exception('Failed to restore database, exitcode: {}'.format(exitcode))
        logging.info("{} was ended with exit code {}".format(args[0], exitcode))

    def restore_database(self, psql_path, dump_path):
        pg_restore_path = os.path.join(psql_path, 'bin/pg_restore')
        psql_path = os.path.join(psql_path, 'bin/psql')
        password = sdk2.Vault.data(self.Parameters.password)
        drop_all_tables_sql = ' '.join([
            '''DO $$ DECLARE r RECORD; BEGIN''',
            '''    SET lock_timeout TO '30s';''',
            '''    FOR r IN (SELECT tablename FROM pg_tables WHERE schemaname = 'public') LOOP''',
            '''        EXECUTE 'DROP TABLE IF EXISTS ' || quote_ident(r.tablename) || ' CASCADE';''',
            '''    END LOOP;''',
            '''END $$''',
        ])
        drop_all_views_sql = ' '.join([
            '''DO $$ DECLARE r RECORD; BEGIN''',
            '''    SET lock_timeout TO '30s';''',
            '''    FOR r IN (SELECT viewname FROM pg_views WHERE schemaname = 'public') LOOP''',
            '''        EXECUTE 'DROP VIEW ' || quote_ident(r.viewname);''',
            '''    END LOOP;''',
            '''END $$''',
        ])
        with sdk2.helpers.ProcessLog(self, logging.getLogger('psql_restore')) as pl:
            for command in (drop_all_tables_sql, drop_all_views_sql):
                self.run_pg_command_with_password(
                    psql_path,
                    self.Parameters.host,
                    self.Parameters.port,
                    self.Parameters.dbname,
                    self.Parameters.username,
                    password,
                    ['--command={}'.format(command)],
                    pl
                )

            host = self._find_host_restore(
                self.Parameters.host,
                self.Parameters.port,
                self.Parameters.dbname,
                self.Parameters.username,
                password,
                psql_path,
                pl
            )
            self.run_restore_with_password(
                [
                    pg_restore_path,
                    '--dbname={}'.format(self.Parameters.dbname),
                    '--host={}'.format(host),
                    '--port={}'.format(self.Parameters.port),
                    '--username={}'.format(self.Parameters.username),
                    '--password',
                    '--single-transaction',
                    '--schema=public',
                    '--clean',
                    '--if-exists',
                    '--no-privileges',
                    '--no-owner',
                    '--exit-on-error',
                    '--verbose',
                    dump_path,
                ],
                password,
                pl
            )
            # self.run_pg_command_with_password(
            #     psql_path,
            #     self.Parameters.host,
            #     self.Parameters.port,
            #     self.Parameters.dbname,
            #     self.Parameters.username,
            #     password,
            #     ['--command=VACUUM VERBOSE ANALYZE;'],
            #     pl
            # )

    def on_execute(self):
        psql_path = str(
            sdk2.ResourceData(
                sdk2.Resource[self.Parameters.postgres_resource]
            ).path
        )
        self.setup_ramdrive()
        dump_path = self.get_dump_path()
        self.restore_database(psql_path, dump_path)
