# -*- coding: utf-8 -*-
import logging
import subprocess
import os

import sandbox.common.types.resource as ctr
from sandbox import sdk2
from sandbox.common.types.notification import Transport
from sandbox.projects.answers import resources
from sandbox.projects.common.arcadia import sdk


class AnswersRunDBMigrations(sdk2.Task):
    """Applies migration to DB"""

    class Parameters(sdk2.Parameters):

        config_conf_sec = sdk2.parameters.YavSecret(
            'Yav secret with "config.conf" files',
            description='App "config.conf" file with "Database" section pointing to the target DB',
            default='sec-01esk399dj19j3arrb3qy6534p',
            required=True
        )
        config_env = sdk2.parameters.String(
            'Environment type',
            default='trunk',
            required=True
        )
        config_conf_sec_key = sdk2.parameters.String(
            'Key with config.conf',
            default='config.conf.trunk',
            required=True
        )
        migrations_arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Svn url for arcadia with database migrations',
            required=True,
            default_value='arcadia:/arc/trunk/arcadia/'
        )
        migrations_path = sdk2.parameters.String(
            'Path to migrations from arcadia',
            required=True,
            default_value='yweb/yasap/answers/database_migrations'
        )
        database_migrator_resource = sdk2.parameters.LastResource(
            '"database_migrator" bin resource',
            resource_type=resources.AnswersDatabaseMigrator,
            state=(ctr.State.READY,),
            required=True,
        )
        target_revision = sdk2.parameters.String(
            'Target revision',
            default='before',
            required=False
        )

    def _send_notification_and_log(self, msg):
        logging.info(msg)
        if self.Parameters.config_env == 'prod':
            self.server.notification(
                body=msg,
                recipients=["ANSWERS"],
                transport=Transport.TELEGRAM
            )

    def _upgrade_next(self, database_migrator_path, pl, label, db):
        cmd = [database_migrator_path, 'status', '--db', db, '--next_rev', label]
        output = subprocess.check_output(cmd, stderr=pl.stderr)
        if output != '-\n':
            cmd = [database_migrator_path, 'upgrade', '--db', db, '{}_next'.format(label)]
            proc = subprocess.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr)
            exitcode = proc.wait()
            if exitcode:
                self._send_notification_and_log('Migration failed, see details in CI')
                raise Exception('Failed "%s" exitcode: %s' % (cmd, exitcode))

            self._send_notification_and_log('{} migration {} finished for db {}'.format(label, output[:-1], db))
            self._upgrade_next(database_migrator_path, pl, label, db)

    def _run_migrations(self, checkout_dir):
        config_conf = self.Parameters.config_conf_sec.data()[self.Parameters.config_conf_sec_key]
        os.chdir(checkout_dir)
        config_path = 'pyhocon_config.conf'
        logging.info('Writing config file to "%s"', config_path)
        with open(config_path, 'w') as f:
            f.write(config_conf)

        database_migrator_path = str(sdk2.ResourceData(self.Parameters.database_migrator_resource).path)
        with sdk2.helpers.ProcessLog(self, logging.getLogger('migrator')) as pl:
            for db in ['main', 'series']:
                if self.Parameters.target_revision != 'after':
                    self._upgrade_next(database_migrator_path, pl, 'before', db)
                if self.Parameters.target_revision != 'before':
                    self._upgrade_next(database_migrator_path, pl, 'after', db)

    def on_execute(self):
        with sdk.mount_arc_path(self.Parameters.migrations_arcadia_url) as mount_path:
            self._send_notification_and_log('Start running migrations')
            logging.info('Mounted directory path {}'.format(mount_path))
            self._run_migrations(os.path.join(mount_path, self.Parameters.migrations_path))
            self._send_notification_and_log('Finish running migrations')
