# -*- coding: utf-8 -*-
import logging

from sandbox import sdk2
from sandbox.common import errors

import sandbox.common.types.resource as ctr
import sandbox.common.types.task as ctt
from sandbox.projects.answers import resources
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.sdk2 import parameters


DEFAULT_TANK_CONFIG = """phantom:
  load_profile: {load_type: rps, schedule: 'line(1,140,5m) const(140,5m)'}
  writelog: proto_warning
autostop:
  autostop: ['http(4xx,30%,5)', 'http(5xx,10%,5)', 'http(0,10%,10)', 'quantile(95,4000ms,30s)']
  enabled: true
  package: yandextank.plugins.Autostop
uploader:
  operator: lunapark
  task: ANSWERS-6906
  job_name:
  job_dsc: 'started from Sandbox'
  meta: {jenkinsbuild: 'http://sandbox.yandex-team.ru/task/{}/view'}
"""


class AnswersStressTestLauncher(sdk2.Task):
    """Prepare config and ammo for SHOOT_VIA_TANKAPI task"""

    class Parameters(sdk2.Parameters):
        nanny_service = parameters.String(
            'Nanny service name (shooting target)',
            default='answers_backend_stress',
            required=True
        )
        ammo_resource = parameters.LastResource(
            'Resource with ammo',
            resource_type=resources.AnswersStressAmmo,
            state=(ctr.State.READY,),
            required=True,
        )
        config_content = sdk2.parameters.String(
            'Tank config contents',
            multiline=True,
            default_value=DEFAULT_TANK_CONFIG,
            required=True
        )

    class Context(sdk2.Task.Context):
        shoot_task_id = None

    def on_execute(self):
        if self.Context.shoot_task_id is None:
            self.create_shoot_task()
        else:
            self.check_shoot_task()

    def create_shoot_task(self):
        shoot_task = ShootViaTankapi(
            self,
            description='Shoot to {}'.format(self.Parameters.nanny_service),
            use_public_tanks=True,
            nanny_service=self.Parameters.nanny_service,
            override_nanny_port=80,
            config_source='file',
            config_content=self.Parameters.config_content,
            ammo_source='resource',
            ammo_resource=self.Parameters.ammo_resource
        )
        self.Context.shoot_task_id = shoot_task.id
        logging.info('Starting shoot_via_tank_api for nanny service {}. Shoot task id {}.'.format(
            self.Parameters.nanny_service,
            shoot_task.id
        ))
        shoot_task.enqueue()
        raise sdk2.WaitTask(
            [shoot_task.id],
            list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True,
            timeout=3600
        )

    def check_shoot_task(self):
        task = sdk2.Task[self.Context.shoot_task_id]
        if task.status != ctt.Status.SUCCESS:
            raise errors.TaskFailure(
                'Subtask is not OK: {task_name}::{id}::{status}'.format(
                    task_name=type(task).name,
                    id=task.id,
                    status=task.status,
                )
            )
