from __future__ import absolute_import

import logging
from sandbox import sdk2

from sandbox.common import errors
from sandbox.common.types.task import Status, ReleaseStatus
from sandbox.projects import DeployNannyDashboard as deploy_nanny_dashboard
from sandbox.projects.collections.mixins import YasmReportable

from sandbox.projects.answers.AnswersTheQuestionSiteMap import AnswersTheQuestionSiteMapTask

from sandbox.projects.answers import resources


NANNY_PARAMS = {
    resources.Environments.TEST: {
        'deployment_nanny_dashboard_name': 'answers_backend_testing',
        'deployment_nanny_dashboard_recipe': 'deploy_thequestion_sitemaps',
        'deployment_nanny_dashboard_filter': 'sitemap',
    },
    resources.Environments.PROD: {
        'deployment_nanny_dashboard_name': 'answers_production',
        'deployment_nanny_dashboard_recipe': 'deploy_thequestion_sitemaps',
        'deployment_nanny_dashboard_filter': 'sitemap',
    },
}


SITEMAP_DEFAULT_REQUIREMENTS = {
    resources.Environments.TEST: {
        "disk_space": 8,
        "ramdrive": 8,
        "ram": 16,
    },
    resources.Environments.PROD: {
        "disk_space": 16,
        "ramdrive": 16,
        "ram": 16,
    }
}


class AnswersTheQuestionSiteMapWorkflow(sdk2.Task, YasmReportable):
    """ Create collections sitemap from rb_torrent and deploy in nanny """
    class Parameters(sdk2.Task.Parameters):
        postgres_resource = sdk2.parameters.Resource(
            'Resource with PSQL',
            resource_type=resources.AnswersPostgresql,
            required=True,
        )
        gpg_key_owner = sdk2.parameters.String(
            'GPG key owner of dump', required=True,
        )
        env = sdk2.parameters.String(
            'Environment that launched this task (env, prod)',
            required=True,
            choices=[
                ('Production', resources.Environments.PROD),
                ('Test', resources.Environments.TEST)
            ],
        )
        dump_host = sdk2.parameters.String(
            'Original host', required=False, default='',
        )
        sitemap_ramdrive = sdk2.parameters.Integer(
            'Ram drive size for ANSWERS_THE_QUESTION_SITEMAP_TASK in Gb',
            default=0,
        )
        sitemap_disk_space = sdk2.parameters.Integer(
            'Disk space for ANSWERS_THE_QUESTION_SITEMAP_TASK in Gb',
            required=True, default=0,
        )
        sitemap_ram = sdk2.parameters.Integer(
            'Ram for ANSWERS_THE_QUESTION_SITEMAP_TASK in Gb',
            required=False, default=0,
        )
        service_name = sdk2.parameters.String(
            'Service name',
            required=False,
            default='',
        )
        monitoring_server_host = sdk2.parameters.String(
            'Monitoring server',
            default='monit.n.yandex-team.ru',
        )

    class Context(sdk2.Task.Context):
        remote_copy_task_id = 0
        sitemap_task_id = 0
        upload_task_id = 0

    def _create_sitemap_task(self):
        if self.Context.sitemap_task_id:
            return

        workflow = self.Parameters.env

        sitemap_task = AnswersTheQuestionSiteMapTask(
            self,
            description='Sitemap with env={}'.format(workflow),
            kill_timeout=6 * 3600,
            create_subtask=False,
            **{
                'use_last_dump': True,
                'postgres_resource': self.Parameters.postgres_resource,
                'gpg_key_owner': self.Parameters.gpg_key_owner,
                'ramdrive_size': (
                    self.Parameters.sitemap_ramdrive or
                    SITEMAP_DEFAULT_REQUIREMENTS[workflow]["ramdrive"]
                ),
                'env': workflow
            }
        )

        execution_space = (
            self.Parameters.sitemap_disk_space or
            SITEMAP_DEFAULT_REQUIREMENTS[workflow]["disk_space"]
        ) << 10
        ram = (
            self.Parameters.sitemap_ram or
            SITEMAP_DEFAULT_REQUIREMENTS[workflow]["ram"]
        ) << 10

        sdk2.Task.server.task[sitemap_task.id].update({
            'requirements': {
                'disk_space': execution_space,
                'ram': ram,
            },
        })
        self.Context.sitemap_task_id = sitemap_task.enqueue().id
        self.Context.save()
        logging.info(
            'Sitemap task created with id %s', self.Context.sitemap_task_id
        )

    def _release_sitemap_task(self):
        task = sdk2.Task[self.Context.sitemap_task_id]
        if task.status in (Status.RELEASING, Status.RELEASED):
            return

        self.server.release({
            'task_id': self.Context.sitemap_task_id,
            'subject': "{}".format(self.Parameters.description),
            'type': {
                resources.Environments.TEST: ReleaseStatus.TESTING,
                resources.Environments.PROD: ReleaseStatus.STABLE,
            }[self.Parameters.env],
        })

    def _create_upload_task(self, ):
        if self.Context.upload_task_id:
            return

        workflow = self.Parameters.env
        deploy_task_params = NANNY_PARAMS[workflow]
        deploy_task_params.update({
            'deployment_task_id': self.Context.sitemap_task_id,
            'deployment_release_status': {
                resources.Environments.TEST: "testing",
                resources.Environments.PROD: "stable",
            }[workflow],
            'vault_name': 'nanny_oauth_token',
            'vault_owner': "YASAP",
            'deployment_nanny_bool_wait': True,
        })

        task_cls = sdk2.Task[deploy_nanny_dashboard.DeployNannyDashboard.type]
        nanny_dash_task = task_cls(
            self,
            description='Upload {workflow} sitemaps for answers.'.format(
                workflow=workflow
            ),
            **deploy_task_params
        )
        self.Context.upload_task_id = nanny_dash_task.enqueue().id
        self.Context.save()
        logging.info(
            'Upload task was created with id %s', self.Context.upload_task_id
        )

    def _check_task(self, task_id, valid_statuses):
        """
        :type task: int
        :type valid_statuses: list
        """
        task = sdk2.Task[task_id]
        if task.status not in valid_statuses:
            raise errors.TaskFailure(
                'Subtask is not OK: {task_name}::{id}::{status}'.format(
                    task_name=type(task).name,
                    id=task.id,
                    status=task.status,
                )
            )

    def _wait_task(self, executing_tasks):
        """
        :type task: int | list[int]
        """
        memoize_name = (
            ','.join([str(i) for i in executing_tasks])
            if isinstance(executing_tasks, list) else
            str(executing_tasks)
        )
        with self.memoize_stage[memoize_name]:
            raise sdk2.WaitTask(
                executing_tasks,
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True,
            )

    def on_execute(self):
        self._create_sitemap_task()
        self._wait_task(self.Context.sitemap_task_id)
        self._check_task(
            self.Context.sitemap_task_id,
            [Status.SUCCESS, Status.RELEASED, Status.RELEASING],
        )

        self._release_sitemap_task()

        self._create_upload_task()
        self._wait_task([
            self.Context.upload_task_id,
        ])

        self._check_task(self.Context.upload_task_id, [Status.SUCCESS])
        self._report_lag(
            "thequestion_{}_sitemap_released".format(self.Parameters.env)
        )
