# -*- coding: utf-8 -*-

from sandbox import sdk2
import logging

from sandbox.common import errors
from sandbox.common.types.task import Status

from sandbox.projects.answers import BuildAnswersBackend
from sandbox.projects.release_machine.tasks import CreateBranchOrTag
from sandbox.projects.release_machine.core import const
from sandbox.projects.common.constants import (
    ARCADIA_URL_KEY,
)


class BuildAnswersBackendEx(sdk2.Task):
    """
        Сборка бандла бинарей бекенда Яндекс.Знатоков
    """
    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(
            'Svn url for arcadia',
            required=True,
            default_value='arcadia:/arc/trunk/arcadia'
        )
        create_branch = sdk2.parameters.Bool(
            'Create branch',
            required=False,
            default=False,
        )
        create_tag = sdk2.parameters.Bool(
            'Create tag',
            required=False,
            default=False,
        )

    def _create_create_branch_or_tag_subtask(self, branching_mode):

        self.create_branch_or_tag_task = CreateBranchOrTag(
            self,
            description='Create branch or tag',
            **{
                'component_name': 'answers_base',
                'branching_mode': branching_mode
            }
        )

        self.Context.create_branch_or_tag_task_id = self.create_branch_or_tag_task.enqueue().id
        self.Context.save()
        logging.info(
            'Create tag or branch task created with id %s', self.Context.create_branch_or_tag_task_id
        )
        return self.create_branch_or_tag_task

    def _create_build_subtask(self, arcadia_url):

        self.build_task = BuildAnswersBackend(
            self,
            description='Build answers backend',
            **{
                ARCADIA_URL_KEY: arcadia_url
            }
        )

        self.Context.build_task_id = self.build_task.enqueue().id
        self.Context.save()
        logging.info(
            'Build answers backend task created with id %s', self.Context.create_branch_task_id
        )
        return self.build_task

    def _check_task(self, task_id, valid_statuses):
        """
        :type task: int
        :type valid_statuses: list
        """
        task = sdk2.Task[task_id]
        if task.status not in valid_statuses:
            raise errors.TaskFailure(
                'Subtask is not OK: {task_name}::{id}::{status}'.format(
                    task_name=type(task).name,
                    id=task.id,
                    status=task.status,
                )
            )

    def _wait_task(self, executing_tasks):
        """
        :type task: int | list[int]
        """
        memoize_name = (
            ','.join([str(i) for i in executing_tasks])
            if isinstance(executing_tasks, list) else
            str(executing_tasks)
        )
        with self.memoize_stage[memoize_name]:
            raise sdk2.WaitTask(
                executing_tasks,
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True,
            )

    def on_execute(self):
        branching_mode = const.BranchingMode.TRUNK

        if self.Parameters.create_branch:
            branching_mode = const.BranchingMode.TRUNK_TO_BRANCH
            # check build url is trunk
        elif self.Parameters.create_tag:
            branching_mode = const.BranchingMode.BRANCH_TO_TAG
            # check build url is branch

        if branching_mode != const.BranchingMode.TRUNK:
            self._create_create_branch_or_tag_subtask(branching_mode)

            self._wait_task(self.Context.create_branch_or_tag_task_id)
            self._check_task(
                self.Context.create_branch_or_tag_task_id,
                [Status.SUCCESS, Status.RELEASED, Status.RELEASING],
            )

            self.Parameters.arcadia_dir = self.create_branch_or_tag_task.Context.result_path

        self._create_build_subtask(self.Parameters.arcadia_url)

        self._wait_task(self.Context.build_task_id)
        self._check_task(
            self.Context.create_branch_or_tag_task_id,
            [Status.SUCCESS, Status.RELEASED, Status.RELEASING],
        )
