# -*- coding: utf-8 -*-
import logging
import os
import time

import sandbox.common.types.client as ctc
import sandbox.projects.common.build.parameters as params
import sandbox.projects.common.arcadia.sdk as sdk
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.release_machine.mixins.build as rm_build_mixin
import sandbox.projects.release_machine.rm_notify as rm_notify
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.common.search.gdb as gdb
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.projects.release_machine.components.configs.base import BaseCfg


class BASEANSWERS:
    """
        Исполняемый файл базового поиска
    """
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True
    releasers = 'kasperovich,dshmatkov,anoh,theigel'
    arcadia_build_path = 'yweb/yasap'
    arcadia_build_name = 'baseanswers'


STRING = 'string'
RESOURCE = 'resource'
LOCAL_RELEASE_DIR = 'release'

build_fields = {
    'baseanswers': BASEANSWERS,
}


@rm_notify.notify2()
class BuildBaseAnswers(rm_build_mixin.ComponentReleaseTemplate, nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Build binaries for basesearch releases without release to nanny and with using arcadia sdk
    """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group(params.BASE_BUILD_GROUP_NAME) as bb_group:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)
            arcadia_patch = sdk2.parameters.String(
                'Apply patch (diff file rbtorrent, paste.y-t.ru link or plain text). Doc: https://nda.ya.ru/3QTTV4',
                default='', multiline=True
            )
            clear_build = sdk2.parameters.Bool('Clear build', default=True)
        with sdk2.parameters.Group(params.BUILD_SYSTEM_GROUP_NAME) as bs_group:
            with sdk2.parameters.String('Build type', required=True) as build_type:
                build_type.values.release = build_type.Value('Release', default=True)
                build_type.values.debug = 'Debug'
                build_type.values.profile = 'Profile'
                build_type.values.coverage = 'Coverage'
                build_type.values.relwithdebinfo = 'Release with debug info'
                build_type.values.valgrind = 'Valgrind (debug)'
                build_type.values.valgrind_release = 'Valgrind (release) (only for build with ya)'

            definition_flags = sdk2.parameters.String(
                'Definition flags (for example "-Dkey1=val1 ... -DkeyN=valN")', required=False
            )

            target_platform_flags = sdk2.parameters.String(
                'Target platform flags (only for cross-compilation)', required=False
            )
            musl = sdk2.parameters.Bool("Use --musl option", default=False)
            checkout = sdk2.parameters.Bool("Set checkout option for build", default=True)
        with sdk2.parameters.Group("Projects") as p_group:
            with sdk2.parameters.CheckGroup('Build projects') as search_projects:
                for field in build_fields:
                    setattr(
                        search_projects.values, field, search_projects.Value('Build {}'.format(field), checked=True)
                    )
        with sdk2.parameters.Group("Target resources") as tr:
            basesearch_res_id = sdk2.parameters.ParentResource(
                "Baseanswers resource from parent task", required=False, do_not_copy=True
            )

        use_ram_drive = sdk2.parameters.Bool("Checkout arcadia to ram drive", default=False)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 40 * 1024
        client_tags = ctc.Tag.SSD

    def on_enqueue(self):
        setattr(self.Context, rm_const.COMPONENT_CTX_KEY, BaseCfg.name)
        if self.Parameters.use_ram_drive:
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 30 * 1024, None)

    def get_targets(self):
        return [
            'contrib/python/celery/py2/bin',
            'yweb/yasap/answers/database_migrations',
            'yweb/yasap/answers',
            'yweb/yasap/answers/worker',
            'yweb/yasap/answers/admin',
        ]

    def on_execute(self):
        # idempotent operation
        gdb.append_to_release_sdk2(self)

        arcadia_src = self.Parameters.checkout_arcadia_from_url
        self.Context.checkout_arcadia_from_url = arcadia_src.strip('/')

        # get arcadia build path
        target_dirs = self.get_targets()
        sdk2.path.Path(os.path.abspath(LOCAL_RELEASE_DIR)).mkdir(mode=0o755, exist_ok=True)

        def_flags = sdk.parse_flags(self.Parameters.definition_flags)
        def_flags['SANDBOX_TASK_ID'] = self.id

        # Checkout useful part of Arcadia to decrease checkout time
        if self.Parameters.use_ram_drive:
            os.chdir(str(self.ramdrive.path))
        do_clone_start = int(time.time())
        arcadia_src_dir = sdk.do_clone(arcadia_src, self, use_checkout=False)
        do_clone_end = int(time.time())
        os.chdir(str(self.path()))
        release_svn_info = sdk2.svn.Arcadia.info(arcadia_src)
        logging.info('Release revision is %s.', release_svn_info['entry_revision'])

        for dependent_part in [
            os.path.join("contrib", "java"),  # 126 Mb
            os.path.join("contrib", "libs"),  # 1.3 Gb
            os.path.join("contrib", "python"),  # 281 Mb
            os.path.join("contrib", "tools"),
            os.path.join("contrib", "celery"),
            "dict",     # 633 Mb
            "kernel",
            "library",  # 198 Mb
            "mapreduce",
            "search",  # 461 Mb
            "tools",  # 136 Mb
            "util",
            "yql",
            "yweb",   # 178 Mb
        ]:
            logging.debug("Update %s dir", dependent_part)
            # use only top level directories
            sdk2.svn.Arcadia.update(
                os.path.join(arcadia_src_dir, dependent_part),
                revision=release_svn_info["entry_revision"],
                set_depth="infinity",
                parents=True
            )
        checkout_end = int(time.time())
        sdk.do_build(
            "ya", arcadia_src_dir,
            target_dirs, self.Parameters.build_type,
            clear_build=True, results_dir=LOCAL_RELEASE_DIR,
            def_flags=def_flags, target_platform_flags=self.Parameters.target_platform_flags,
            checkout=self.Parameters.checkout, musl=self.Parameters.musl
        )
        build_end = int(time.time())
        self.set_info("Do clone time is {} sec.".format(do_clone_end - do_clone_start))
        self.set_info("Checkout time is {} sec.".format(checkout_end - do_clone_end))
        self.set_info("Build time is {} sec.".format(build_end - checkout_end))
        self.Context.do_clone_time = do_clone_end - do_clone_start
        self.Context.checkout_time = checkout_end - do_clone_end
        self.Context.build_time = build_end - checkout_end
