# coding: utf-8

import os
import time
from sandbox import common
from sandbox import sdk2

import sandbox.common.types.task as ctt

from sandbox.projects.common.nanny import nanny
from sandbox.projects.answers import resources
from sandbox.sandboxsdk import paths
from sandbox.sdk2.helpers import subprocess as sp


class BuildAnswersRecommenderData(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        # take shard builder binary
        index_builder_res = sdk2.parameters.Resource(
            "Recommender index builder binary",
            resource_type=resources.AnswersRecommenderIndexBuilder,
            required=True
        )
        # take DSSM language model
        dssm_model_res = sdk2.parameters.Resource(
            "DSSM model",
            required=True
        )
        # MatrixNet model for interesting
        interesting_mx_model_res = sdk2.parameters.Resource(
            "Interesting MatrixNet model",
            required=False
        )
        # MatrixNet model for serp_availability
        serp_availability_mx_model_res = sdk2.parameters.Resource(
            "SerpAvailability MatrixNet model",
            required=False
        )
        yt_proxy = sdk2.parameters.String('YT proxy', required=True, default='arnold')
        yt_token_vault = sdk2.parameters.String('YT token vault', required=True)
        yt_dumps_dir = sdk2.parameters.String("YT dump dir", required=True)
        yt_prediction_results_table = sdk2.parameters.String("Prediction results table on YT", required=False)
        history_table = sdk2.parameters.String("Table with user activity history", required=True)
        query_table = sdk2.parameters.String("Table with web query frequencies for Wordstat", required=False)
        release_type = sdk2.parameters.String("Release type", required=False)
        include_dj_shard = sdk2.parameters.Bool("Download DJ shard for prestable data.", default=True)

    def on_execute(self):
        index_builder_path = str(sdk2.ResourceData(self.Parameters.index_builder_res).path)
        dssm_model_path = str(sdk2.ResourceData(self.Parameters.dssm_model_res).path)
        shard_dir = self.path("shard")
        if os.path.exists(str(shard_dir)):
            paths.remove_path(shard_dir)
        paths.make_folder(shard_dir, True)

        env = os.environ.copy()
        env["YT_TOKEN"] = sdk2.Vault.data(self.Parameters.yt_token_vault)
        env["OMP_NUM_THREADS"] = "16"

        with sdk2.helpers.ProcessLog(self, logger="index_builder_log") as pl:
            params = [
                index_builder_path + "/index_builder/recommender_index_builder",
                "--yt-cluster", self.Parameters.yt_proxy,
                "--yt-folder", self.Parameters.yt_dumps_dir,
                "--predicted-values-yt-table", self.Parameters.yt_prediction_results_table,
                "--index-folder", str(shard_dir),
                "--dssm-model", dssm_model_path,
                "--history-table", self.Parameters.history_table,
            ]
            if self.Parameters.interesting_mx_model_res:
                interesting_mx_model_path = str(sdk2.ResourceData(self.Parameters.interesting_mx_model_res).path)
                params += ["--mn-interesting", interesting_mx_model_path]
            if self.Parameters.serp_availability_mx_model_res:
                serp_availability_mx_model_path = str(sdk2.ResourceData(self.Parameters.serp_availability_mx_model_res).path)
                params += ["--mn-serp-availability", serp_availability_mx_model_path]
            if self.Parameters.query_table:
                params += ["--web-query-frequencies-yt-table", self.Parameters.query_table]

            ret_code = sp.Popen(params, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret_code != 0:
                raise common.errors.TaskFailure('index_builder run failed')

        shard_archive_name = "shard.tar.gz"
        with sdk2.helpers.ProcessLog(self, logger="pack_to_tar_gz") as pl:
            sp.Popen(
                [
                    "tar",
                    "-czvf",
                    shard_archive_name,
                    "-C",
                    str(shard_dir),
                    "."
                ],
                stdout=pl.stdout, stderr=sp.STDOUT
            ).wait()

        sdk2.ResourceData(resources.AnswersRecommenderData(
            self, "Recommender index data", shard_archive_name
        )).ready()

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        release_type_map = {
            "stable": ctt.ReleaseStatus.STABLE,
            "prestable": ctt.ReleaseStatus.PRESTABLE,
            "testing": ctt.ReleaseStatus.TESTING,
        }
        if self.Parameters.release_type is not None:
            nanny.ReleaseToNannyTask2.on_release(self, dict(
                releaser=self.author,
                release_status=release_type_map.get(self.Parameters.release_type, ctt.ReleaseStatus.UNSTABLE),
                release_subject="Recommender data",
                email_notifications=dict(to=[], cc=[]),
                release_comments="Recommender data",
            ))
        if self.Parameters.release_type == "stable":
            time.sleep(300)
            nanny_client = nanny.NannyClient(
                api_url='http://nanny.yandex-team.ru/',
                oauth_token=sdk2.Vault.data('ANSWERS', 'answers_nanny_oauth_token')
            )
            nanny_client.deploy_dashboard(
                dashboard='answers_production',
                recipe='answers_recommender_production_recipe',
                use_latest_snapshot=True
            )
