# -*- coding: utf-8 -*-

import logging
import os

import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox.projects.answers import resources
from sandbox.sdk2.helpers import ProcessLog, subprocess


# TODO make config generator depends on env
YAUTH_CONFIG_SECTION = '''
Yauth {{
    JwtSecretForTesting: "{jwt_secret}"
    BlackboxUrl: "http://blackbox-mimino.yandex.net/blackbox"
}}
'''

PROD_CONFIG_DATA = YAUTH_CONFIG_SECTION + '''
Database {{
    User: "robot_answers"
    Password: "{psql_password}"
    Name: "answers_production_db_pg"
    Hosts {{
        Host: "vla-1x56dtb0zz2nshdl.db.yandex.net"
        Port: 6432
        DC: "vla"
    }}
    Hosts {{
        Host: "vla-wafka366qzf1jff3.db.yandex.net"
        Port: 6432
        DC: "vla"
    }}
    Hosts {{
        Host: "sas-dfbnf56dt476y61r.db.yandex.net"
        Port: 6432
        DC: "sas"
    }}
    Hosts {{
        Host: "sas-ii4azrljbdfpgqzq.db.yandex.net"
        Port: 6432
        DC: "sas"
    }}
    Hosts {{
        Host: "man-g3jetrx9jik56yji.db.yandex.net"
        Port: 6432
        DC: "man"
    }}
    Hosts {{
        Host: "man-vu761wgvfvqs7yed.db.yandex.net"
        Port: 6432
        DC: "man"
    }}
}}
'''

PRESTABLE_CONFIG_DATA = YAUTH_CONFIG_SECTION + '''
Database {{
    User: "robot_answers"
    Name: "answers_prestable_db_pg"
    Password: "{psql_password}"
    Hosts {{
        Host: "vla-524j0sg07jxeojwq.db.yandex.net"
        Port: 6432
        DC: "vla"
    }}
    Hosts {{
        Host: "sas-ly9izw54q30b24c1.db.yandex.net"
        Port: 6432
        DC: "sas"
    }}
    Hosts {{
        Host: "man-zagt968knk2u6d3l.db.yandex.net"
        Port: 6432
        DC: "man"
    }}
}}
'''


class BuildAnswersStressAmmo(sdk2.Task):
    ''' Build stress ammo file '''

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 3600

        password = sdk2.parameters.String(
            'Database user password secret', required=True
        )

        yql_token = sdk2.parameters.String(
            'Yql token', required=True
        )

        jwt_secret = sdk2.parameters.String(
            'Jwt secret', required=True
        )

        env = sdk2.parameters.String(
            'Database environment',
            choices=[
                ('prod', resources.Environments.PROD),
                ('prestable', resources.Environments.PRESTABLE),
            ],
            required=True,
        )

        use_last_ctl_binary = sdk2.parameters.Bool(
            'Use last answers ctl binary',
            required=True,
            default=True,
        )
        with use_last_ctl_binary.value[False]:
            ctl_binary = sdk2.parameters.Resource(
                'Answers ctl binary',
                resource_type=resources.AnswersCtl,
                required=True,
            )

    def _make_config_resource(self):
        if self.Parameters.env == resources.Environments.PROD:
            config_template = PROD_CONFIG_DATA
        elif self.Parameters.env == resources.Environments.PRESTABLE:
            config_template = PRESTABLE_CONFIG_DATA
        else:
            raise Exception('Unknown env param')

        config = config_template.format(
            jwt_secret=sdk2.Vault.data(self.Parameters.jwt_secret),
            psql_password=sdk2.Vault.data(self.Parameters.password)
        )

        answers_config_resource = sdk2.ResourceData(resources.AnswersConfig(
            self, 'Answers config file', resources.AnswersConfig.filename
        ))
        answers_config_resource.path.write_bytes(config)
        answers_config_resource.ready()

    def _get_ctl_binary_path(self):
        if not self.Parameters.use_last_ctl_binary:
            ctl_path = str(
                sdk2.ResourceData(
                    sdk2.Resource[self.Parameters.ctl_binary]
                ).path
            )
        else:
            last_ctl_resource = sdk2.Resource.find(
                resources.AnswersCtl,
                state=ctr.State.READY,
            ).order(
                -resources.AnswersCtl.id,
            ).first()
            ctl_path = str(sdk2.ResourceData(last_ctl_resource).path)
        return ctl_path

    def on_execute(self):
        self._make_config_resource()
        logging.info('Answers {env} config created'.format(env=self.Parameters.env))

        ctl_binary_path = self._get_ctl_binary_path()
        logging.info('Using ctl resource from: %s', ctl_binary_path)

        cmd_parameters = '{production_pass} make_stress_ammo {output_filename} --yql_token {yql_token}'.format(
            production_pass='--i_solemnly_swear_that_i_am_up_to_no_good',
            output_filename=resources.AnswersStressAmmo.filename,
            yql_token=sdk2.Vault.data(self.Parameters.yql_token)
        )
        logging.info('Prepared command line parameters: ' + cmd_parameters)

        with ProcessLog(self, logger=logging.getLogger('debug_log')) as pl:
            subprocess.Popen(ctl_binary_path + ' ' + cmd_parameters,
                             shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT).wait()
        logging.info('ctl execution finished')

        stress_ammo_resource = sdk2.ResourceData(resources.AnswersStressAmmo(
            self,
            'Answers stress ammo resource',
            os.path.join(os.getcwd(), resources.AnswersStressAmmo.filename)
        ))
        stress_ammo_resource.ready()

        logging.info('Stress ammo created as type {} and filename: {}'.format(
            resources.AnswersCtl,
            resources.AnswersStressAmmo.filename
        ))
