# coding: utf-8

import datetime
import os
from sandbox import sdk2
from sandbox.projects.answers import resources
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.answers.common.utils import get_last_dump_dir
from sandbox.sdk2 import parameters

FILENAME = 'suggest.tsv'

USERS_TABLENAME = 'answers__users'
USERS_STATS_TABLENAME = 'answers__user_stats'

TEMPLATE = ('{query}\t{result}\t{weight}\t' + chr(0xA))


# XXX: copied from code because it cant be used in sandbox


def get_verified_level(trust_level):
    return {
        0: 0,
        1: 0,
        2: 1,
        3: 0,
        4: 1,
        5: 2,
        6: 2
    }[trust_level]


class BuildAnswersUsersSuggestions(sdk2.Task):
    class Parameters(sdk2.Parameters):
        yt_dumps_directory = parameters.String('YT Directory With Dumps', required=True)
        yt_cluster = parameters.String('YT Cluster', required=True, default='banach')
        yt_token_secret = parameters.String('YT Token Secret', required=True)
        yql_token_secret = parameters.String('YQL Token Secret', required=True)
        env = parameters.String(
            'Database environment',
            choices=[
                ('dev', resources.Environments.DEV),
                ('prod', resources.Environments.PROD),
                ('prestable', resources.Environments.PRESTABLE),
            ],
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yql'),
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
            PipEnvironment('marshmallow'),
        )

    def get_yt_client(self):
        from yt.wrapper.client import YtClient
        token = sdk2.Vault.data(self.Parameters.yt_token_secret)
        yt_client = YtClient(proxy=self.Parameters.yt_cluster, token=token)
        return yt_client

    def get_yql_client(self):
        from yql.api.v1.client import YqlClient
        token = sdk2.Vault.data(self.Parameters.yql_token_secret)
        yql_client = YqlClient(db=self.Parameters.yt_cluster, token=token)
        return yql_client

    def temp_rows_gen(self, rows):
        from projects.answers.BuildAnswersUsersSuggestions.schema import UserSchema
        schema = UserSchema()

        for row in rows:
            if row['status'] != 0:
                continue

            row = dict(row)
            trust_level = row['trust_level']
            weight = trust_level + 1
            row['trust_level'] = get_verified_level(trust_level)

            yield {
                'query': ':'.join([row['display_name'], row['id']]),
                'result': schema.dumps(row).data,
                'weight': weight
            }

            if row['show_login'] and row['login'].lower() != row['display_name'].lower():
                yield {
                    'query': row['login'],
                    'result': schema.dumps(row).data,
                    'weight': weight
                }

    def get_data(self, yt_client, yql_client, temp_table_path):
        dump_dir = get_last_dump_dir(yt_client, self.Parameters.yt_dumps_directory)

        query = '''
            PRAGMA SimpleColumns;

            SELECT users.* FROM `{users_table}` as users JOIN `{user_stats_table}` as stats ON users.id == stats.user_id
            WHERE users.status == 0 AND NOT COALESCE(users.is_thequestion, FALSE) AND (stats.questions_count > 0 OR stats.answers_count > 0 OR users.trust_level > 1)
            AND users.id IN (

            (SELECT id FROM `{users_table}` WHERE Yson::ConvertToString(user_type) == 'passport' AND login NOT IN
            (SELECT login FROM `{users_table}` WHERE Yson::ConvertToString(user_type) == 'virtual'))
            UNION ALL (SELECT id FROM `{users_table}` WHERE Yson::ConvertToString(user_type) == 'virtual')

            );
        '''.format(
            users_table=os.path.join(dump_dir, USERS_TABLENAME),
            user_stats_table=os.path.join(dump_dir, USERS_STATS_TABLENAME)
        )
        request = yql_client.query(query, syntax_version=1)
        request.run()

        table = list(request.get_results())[0]

        yt_client.write_table(
            yt_client.TablePath(temp_table_path, append=True),
            self.temp_rows_gen(table.full_dataframe.to_dict(orient='records')),
            raw=False,
        )
        yt_client.run_sort(temp_table_path, sort_by='query')
        result = yt_client.read_table(temp_table_path)
        return result

    def prepare_field(self, str):
        return str.strip().replace('\t', ' ').replace('\n', '')

    def on_execute(self):
        yt_client = self.get_yt_client()
        yql_client = self.get_yql_client()

        with yt_client.TempTable() as temp_table:
            data_gen = self.get_data(yt_client, yql_client, temp_table)

            result_path = os.path.join(os.getcwd(), FILENAME)
            with open(result_path, 'w') as f:
                for item in data_gen:
                    s = TEMPLATE.format(
                        query=self.prepare_field(item['query']),
                        result=self.prepare_field(item['result']),
                        weight=item['weight']
                    )
                    f.write(s)

        resource_metadata = resources.AnswersSuggestUsersData(
            self,
            "Znatoki users suggest map at {}".format(
                datetime.datetime.utcnow().isoformat()
            ),
            result_path,
            env=self.Parameters.env,
        )
        resource_data = sdk2.ResourceData(resource_metadata)
        resource_data.ready()
