# -*- coding: utf-8 -*-
from __future__ import unicode_literals, absolute_import

import logging
import os
import random
from sandbox import sdk2
import shutil
import signal
import socket
import string
import time
import uuid
from sandbox.sdk2.helpers import subprocess


from functools import wraps

MAX_START_RETRIES = 3

MAX_START_WAIT_TIME = 60
START_RETRY_JITTER = 5

MAX_CHECK_STARTED_RETRIES = 3
RETRY_DELAY = 1

LENGTH = 10


class Retry(Exception):
    pass


def generate_random_string(length, use_numbers=False):
    charset = string.ascii_lowercase + string.ascii_uppercase
    if use_numbers:
        charset += string.digits

    result = ''.join(random.choice(charset) for _ in range(length))
    return result


class Config(object):
    def __init__(
            self,
            port=None, keep_work_dir=False, work_dir='',
            username=None, password=None, dbname=None
    ):
        if port is None:
            sock = socket.socket()
            sock.bind(('', 0))
            self.port = sock.getsockname()[1]
        else:
            self.port = port
        self.keep_work_dir = keep_work_dir
        self.work_dir = work_dir
        self.username = 'postgres'
        self.password = 'postgres'
        self.dbname = 'postgres'

    def conn_str(self):
        return 'postgresql + psycopg2: // {user}:{password} @ {host}:{port} / {database}'.format(
            user=self.username,
            password=self.password,
            host='localhost',
            port=self.port,
            database=self.dbname
        )


class PostgreSQLStuff(object):
    def __init__(self, config=None, work_path=None):
        self.logger = logging.getLogger()
        self.config = config or Config()
        self.instance_id = str(uuid.uuid4())
        self.work_dir = self.config.work_dir
        self.is_running = False

        self.db_path = work_path
        self.data_path = os.path.join(self.work_dir, 'data')
        self.pg_ctl_local_path = os.path.join(self.db_path, 'bin/pg_ctl')
        self.postgres_pid_file = os.path.join(self.data_path, 'postmaster.pid')

        self.user_creator = os.path.join(self.db_path, 'bin/createuser')
        self.db_creator = os.path.join(self.db_path, 'bin/createdb')

        self._prepare_files()
        self._init_data_dir()

    def _prepare_files(self):
        self.logger.debug('Preparing working directories')

        if os.path.exists(self.work_dir):
            shutil.rmtree(self.work_dir)
        os.mkdir(self.work_dir)

        if os.path.exists(self.data_path):
            shutil.rmtree(self.data_path)
        os.mkdir(self.data_path)

    def _init_data_dir(self):
        self.logger.debug('Init data directory')
        cmd = [
            self.pg_ctl_local_path,
            'initdb',
            '-o', '--auth-local=trust',
            '-o', '--username={}'.format(self.config.username),
            '-D', self.data_path,
        ]
        self.logger.debug(' '.join(cmd))
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('psql_initdb')
        ) as pl:
            init = subprocess.Popen(cmd)
            exitcode = init.wait()
            if exitcode:
                pl.logger.warning(
                    'Errors while database init.\n%s', exitcode, exc_info=True
                )
                raise Exception('Database init failed')

    def _timing(method):
        @wraps(method)
        def wrap(self, *args, **kwargs):
            start_time = time.time()
            ret = method(self, *args, **kwargs)
            finish_time = time.time()
            self.logger.debug(
                '%s time: %f', method.__name__, finish_time - start_time
            )
            return ret
        return wrap

    def _status(self):
        self.logger.debug('Checking db status')
        cmd = [
            self.pg_ctl_local_path,
            'status',
            '-D', self.data_path,
        ]
        self.logger.debug(' '.join(cmd))
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('status')
        ) as pl:
            check = subprocess.Popen(cmd)
            exitcode = check.wait()
            if exitcode:
                self.is_running = False
                pl.logger.warning(
                    'Db is not launched:\n%s', exitcode, exc_info=True,
                )
                raise Exception('Db is not launched')
        self.is_running = True

    def _start(self):
        self.logger.debug('Try to start local psql with id=%s', self.instance_id)
        cmd = [
            self.pg_ctl_local_path,
            'start',
            '-o', '--port={}'.format(self.config.port),
            '-D', self.data_path,
        ]
        self.logger.debug(' '.join(cmd))
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('psql_start')
        ) as pl:
            start = subprocess.Popen(cmd)
            exitcode = start.wait()
            if exitcode:
                pl.logger.warning(
                    'Failed to start local psql:\n%s',
                    exitcode,
                    exc_info=True
                )
                self._kill()
                raise Retry()

    def _start_wrapper(self):
        self._start()
        time.sleep(1)
        self._status()

    @_timing
    def start_local(self):
        from retry.api import retry_call
        retry_call(
            self._start_wrapper,
            tries=MAX_START_RETRIES,
            delay=RETRY_DELAY,
            max_delay=MAX_START_WAIT_TIME,
            jitter=START_RETRY_JITTER,
            logger=self.logger,
        )

    def _kill(self):
        for pid in self._get_pids():
            try:
                os.kill(pid, signal.SIGKILL)
            except OSError:
                pass

    def _get_pids(self):
        if not os.path.exists(self.postgres_pid_file):
            return []
        with open(self.postgres_pid_file, 'r') as pids:
            pid = pids.readline()
            return [int(pid)]

    def _stop_psql(self):
        self.logger.debug('Stop db instance')
        cmd = [
            self.pg_ctl_local_path,
            'stop',
            '-D', self.data_path,
        ]
        self.logger.debug(' '.join(cmd))
        with sdk2.helpers.ProcessLog(
                self, logging.getLogger('psql_dump')
        ) as pl:
            stop = subprocess.Popen(cmd)
            exitcode = stop.wait()
            if exitcode:
                pl.logger.warning(
                    'Errors while stopping local psql:\n%s',
                    exitcode,
                    exc_info=True
                )
                raise Exception('Errors while stopping local psql')

    @_timing
    def stop_local(self):
        if self.is_running:
            self._stop_psql()
            self._kill()
            self.is_running = False
        if not self.config.keep_work_dir:
            shutil.rmtree(self.work_dir, ignore_errors=True)


class PsqlContextWrapper(object):
    def __init__(self, psql):
        self.psql = psql

    def __enter__(self):
        self.psql.start_local()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.psql.stop_local()
