# -*- coding: utf-8 -*-
import os
import logging

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.resource as ctr
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.answers.resources import AnswersPostgresql
from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


class AntiadblockPgDump(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
        )

    class Parameters(sdk2.Parameters):
        dbname = parameters.String('Database Name', required=True, default='aab_admin_yandex_team_ru_db')
        host = parameters.String('Database Host', required=True, default='sas-w9jnn4x6osfkn7nh.db.yandex.net')
        port = parameters.Integer('Database Port', required=True, default=6432)
        username = parameters.String('Database User', required=True, default='antiadb')
        dbpassword_vault = parameters.String('Database password (vault)', required=True, default='ANTIADB_DB_PASSWORD')
        postgres_resource = parameters.Resource("Resource with PSQL", resource_type=AnswersPostgresql)
        ramdrive_size = parameters.Integer('RamDrive size in GB', default=1)

        yt_token_vault = parameters.String('YT token (vault)', required=True, default='ANTIADBLOCK_YT_LOCKE_TOKEN')
        yt_base_directory = parameters.String('Yt base directory', required=True, default='//home/antiadblock/configs_api/pg_db_dump')
        yt_proxy = parameters.String('Yt proxy', required=True, default='LOCKE')

    def on_create(self):
        self.Parameters.postgres_resource = AnswersPostgresql.find(
            status=ctr.State.READY,
            attrs=dict(arch='linux')
        ).first().id

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        if self.Parameters.ramdrive_size:
            self.Requirements.ramdrive = ctm.RamDrive(
                ctm.RamDriveType.TMPFS,
                int(self.Parameters.ramdrive_size) << 10,
                None
            )

    def setup_ramdrive(self):
        if self.ramdrive:
            logging.info(
                'Setup RamDrive size: {} path: {}'.format(common.utils.size2str(self.ramdrive.size << 20), self.ramdrive.path)
            )
            os.chdir(str(self.ramdrive.path))

    def dump_database(self, psql_path):
        pg_dump_path = os.path.join(psql_path, 'bin/pg_dump')
        dump_path = os.path.join(os.getcwd(), 'dump.prod')
        with sdk2.helpers.ProcessLog(self, logging.getLogger('psql_dump')) as pl:
            psql_password = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()[self.Parameters.dbpassword_vault]
            dump = subprocess.Popen(
                [
                    pg_dump_path,
                    '--dbname={}'.format(self.Parameters.dbname),
                    '--host={}'.format(self.Parameters.host),
                    '--port={}'.format(self.Parameters.port),
                    '--username={}'.format(self.Parameters.username),
                    '--no-owner',
                    '--format=custom',
                    '--schema=public',
                    '--exclude-table={}'.format('repl_mon'),
                    '--file={}'.format(dump_path)

                ],
                stdin=subprocess.PIPE,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            dump.communicate(psql_password)
            exitcode = dump.wait()
        if not os.path.exists(dump_path):
            raise Exception('Dump failed with exitcode {}'.format(exitcode))
        return dump_path

    def write_to_yt(self, dump_path, yt):
        yt_dump_path = os.path.join(self.Parameters.yt_base_directory, 'prod.dump')
        try:
            yt.smart_upload_file(dump_path, yt_dump_path, 'prod.dump', 'replace')
        except Exception as e:
            logging.error('Error. Cannot upload file {}'.format(yt_dump_path))
            logging.error(e.message)
            raise
        else:
            logging.info('Uploads {} to {}'.format(dump_path, yt_dump_path))

    def on_execute(self):
        from yt.wrapper import YtClient
        local_psql_path = str(
            sdk2.ResourceData(
                sdk2.Resource[self.Parameters.postgres_resource]
            ).path
        )
        yt_token = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()[self.Parameters.yt_token_vault]
        self.setup_ramdrive()

        yt = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
        if not yt.exists(self.Parameters.yt_base_directory):
            yt.create('map_node', self.Parameters.yt_base_directory, recursive=True)
        dump_path = self.dump_database(local_psql_path)
        self.write_to_yt(dump_path, yt)
        os.remove(dump_path)
