# -*- coding: utf-8 -*-
import json
import logging
from time import sleep

import sandbox.common.types.client as ctc
from sandbox.common.types.misc import DnsType
from sandbox import sdk2, common

from sandbox.projects.qafw.selenoid import Selenoid
from sandbox.projects.antiadblock.aab_argus_utils import SELENOID_CONFIG


class AntiadblockSelenoidConfig(sdk2.Resource):
    releasable = True
    any_arch = True
    auto_backup = True
    group = 'ANTIADBLOCK'
    releasers = ['ANTIADBLOCK']
    release_subscribers = ["ANTIADBLOCK"]


class Antiadblock_Selenoid_Basic_Executor(sdk2.Task):
    """ Run Selenoid In Subtasks and Run checks"""

    class Requirements(sdk2.Requirements):
        ram = 4 << 10
        dns = DnsType.DNS64  # for external interactions
        privileged = True  # to run apt-get installs in build-all script
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Selenoid parameters") as selenoid_params:
            run_selenoid_as_subtask = sdk2.parameters.Bool('Run the selenoid as a subtask (True) or use quote "selenium'
                                                           '" (False)', default=False)
            with run_selenoid_as_subtask.value[True]:
                one_subtask_per_browser = sdk2.parameters.Bool('If selenoid as a subtask enabled we starts one browser '
                                                               'per subtask', default=False)
            browsers_config_for_selenoid_subtask =\
                sdk2.parameters.String('Browsers json for selenoid subtask',
                                       default=json.dumps(SELENOID_CONFIG),
                                       multiline=True)
        task_kill_timeout = sdk2.parameters.Integer('Timeout in minutes after that task will be kill by sandbox',
                                                    default=80 * 60)

    def on_execute(self):
        self.on_pre_execute()
        selenoid_config = json.loads(self.Parameters.browsers_config_for_selenoid_subtask)
        browsers_config = {}

        if self.Parameters.run_selenoid_as_subtask:
            tasks_ids = {}
            # если не бьем по 1 сабтаска = 1 бро, то это дефолтный конфиг (браузер all)
            browser_jsons = [('all', self.Parameters.browsers_config_for_selenoid_subtask)]

            if self.Parameters.one_subtask_per_browser:
                browser_jsons = [(k, json.dumps({v.pop("selenium_browser_name"): v}))
                                 for k, v in selenoid_config.items()]

            with self.memoize_stage['creating_selenoid_tasks']:
                for bro, browser_json in browser_jsons:
                    child = Selenoid(self,
                                     description="Child of {} with {} browser(s)".format(self.id, bro),
                                     owner=self.owner,
                                     browsers=browser_json,
                                     session_attempt_timeout="240s",
                                     service_startup_timeout="240s",
                                     use_robot_qafw=False,
                                     kill_timeout=self.Parameters.task_kill_timeout,
                                     )
                    child.Requirements.cores = 8
                    child.Requirements.ram = 12 << 10
                    child.save()
                    child.enqueue()

                    # общий список {бро: id}
                    tasks_ids[bro] = child.id

                # сохраним все id сабтасок в контекст
                self.Context.tasks_ids = tasks_ids
                # и будем ждать пока все поднимутся и отдадут свой эндпоинт
                wait_parameters = {}
                for bro, task_id in tasks_ids.items():
                    wait_parameters[task_id] = 'endpoint'
                raise sdk2.WaitOutput(wait_parameters, wait_all=True, timeout=720)

            # восстанавливаем значение после возвращения в таску
            tasks_ids = self.Context.tasks_ids
            # и собираем все эндпоинты для передачи скрипту проверки
            browsers_config = self._collect_all_endpoints(tasks_ids.copy())

        # получаем эндпоинт таски и начинаем проверку
        with self.memoize_stage['run_check']:
            self.execute_code(browsers_config, selenoid_config)

    def on_pre_execute(self):
        pass

    def on_finish(self, prev_status, status):
        self._stop_all_child_tasks()

    def on_break(self, prev_status, status):
        self._stop_all_child_tasks()

    def _collect_all_endpoints(self, tasks_ids):
        wait_timeout = 60  # сендбокс сказал что все таски подняты и параметры заполнены, минуты на сбор их должно хватить
        step_wait = 10  # сек
        browsers_config = {}

        while wait_timeout > 0:
            for bro, task_id in tasks_ids.items():
                endpoint = self._get_selenoid_task(task_id).Parameters.endpoint
                logging.info('Task ID: {}, endpoint: {}'.format(task_id, endpoint))  # иногда не можем получить эндпоинт с Ябро, залогируем тут на случай разбора этой проблемы
                if endpoint:
                    browsers_config[bro] = endpoint

            for bro in browsers_config:  # чтоб не итерироваться по этим бро снова
                tasks_ids.pop(bro, None)

            if not tasks_ids:  # если со всех получили эндпоинты
                break
            sleep(step_wait)
            wait_timeout -= step_wait
        else:  # если за таймаут не получаем всех значений
            raise common.errors.TaskFailure("Failed to get endpoints from tasks {}".format(', '.join(tasks_ids)))

        return browsers_config

    def _get_selenoid_task(self, id):
        return self.find(Selenoid, id=id).first()

    def _stop_all_child_tasks(self):
        # проходимся по всем и завершаем их
        if self.Parameters.run_selenoid_as_subtask and self.Context.tasks_ids:
            for bro, task_id in self.Context.tasks_ids.items():
                self._get_selenoid_task(task_id).stop()

    def execute_code(self, browsers_config, selenoid_config):
        """ Код который необходимо выполнить после поднятия всех браузеров.
            Скорее всего выполняет какую-то проверку на них
            :param browsers_config: json конфиг {bro: endpoint, ...} если браузер==all значит все браузеры на одном эндпоинте
        """
        pass
