# -*- coding: utf-8 -*-
import os
import json

import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.sdk2.helpers import subprocess as sp
from sandbox import sdk2, common

from sandbox.projects.antiadblock.aab_selenoid_basic_executor import Antiadblock_Selenoid_Basic_Executor
from sandbox.projects.antiadblock.aab_automerge_ext import AdblockplusChromeExtension, AdblockplusFirefoxExtension, \
    AdguardChromeExtension, AdguardFirefoxExtension, UblockChromeExtension, UblockFirefoxExtension, \
    AdblockFirefoxExtension, AdblockChromeExtension
from sandbox.projects.antiadblock.aab_argus_utils import get_extension_http_url, get_latest_resource
from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


class AntiadblockDetectCheckerBin(sdk2.Resource):
    releasable = True
    any_arch = True
    auto_backup = True
    group = 'ANTIADBLOCK'
    releasers = ['ANTIADBLOCK']
    release_subscribers = ["ANTIADBLOCK"]


class AntiadblockDetectCheckerProfile(AntiadblockDetectCheckerBin):
    pass


class Antiadblock_Detect_Checker(Antiadblock_Selenoid_Basic_Executor):
    """ Run adb_detect_checker downloaded from svn"""

    class Requirements(Antiadblock_Selenoid_Basic_Executor.Requirements):
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='AAB_DETECT_CHECKER', weight=1, capacity=1)
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH, ctt.Status.Group.WAIT)
        )

    class Parameters(Antiadblock_Selenoid_Basic_Executor.Parameters):
        continue_previous = sdk2.parameters.Bool('Continue writing to previous table, if it exists', default=True)
        without_stat = sdk2.parameters.Bool('Do not use stat to save results, only print results to log', default=False)
        env_vars = sdk2.parameters.String('Optional environment variables (expecting "var1=val1;var2=val2")',
                                          required=False)
        binary = sdk2.parameters.LastReleasedResource(
            'Resource type to run',
            state=(ctr.State.READY,),
            required=False,
            attrs=dict(released='stable'),
        )

    def on_pre_execute(self):
        if not self.Parameters.continue_previous:
            return
        cmd = [str(self.get_bin_res().path), '--has-cases-to-run']
        env = os.environ.copy()
        env["STAT_TOKEN"] = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()["ANTIADBLOCK_STAT_TOKEN"]
        env["EXTENSIONS"] = json.dumps(dict(adblocks=self.get_adblock_extensions_path()))
        with sdk2.helpers.ProcessLog(self, logger="run_pre_execute_log") as log:
            output = sp.check_output(cmd, env=env, stderr=log.stderr)
        if 'true' not in output:
            raise common.errors.TaskStop("no checks to run")

    @staticmethod
    def get_adblock_extensions_path():
        return dict(
            ADBLOCK_CHROME=get_extension_http_url(AdblockChromeExtension),
            ADBLOCK_FIREFOX=get_extension_http_url(AdblockFirefoxExtension),
            ADBLOCK_PLUS_CHROME=get_extension_http_url(AdblockplusChromeExtension),
            ADBLOCK_PLUS_FIREFOX=get_extension_http_url(AdblockplusFirefoxExtension),
            ADGUARD_CHROME=get_extension_http_url(AdguardChromeExtension),
            ADGUARD_FIREFOX=get_extension_http_url(AdguardFirefoxExtension),
            UBLOCK_ORIGIN_CHROME=get_extension_http_url(UblockChromeExtension),
            UBLOCK_ORIGIN_FIREFOX=get_extension_http_url(UblockFirefoxExtension),
        )

    def execute_code(self, browsers_config, selenoid_config):
        bin_res = self.get_bin_res()

        with open(str(sdk2.ResourceData(get_latest_resource(AntiadblockDetectCheckerProfile)).path)) as fp:
            last_released_profile = json.load(fp)

        env = os.environ.copy()
        env["CONTINUE_PREVIOUS"] = str(self.Parameters.continue_previous)
        env["STAT_TOKEN"] = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()["ANTIADBLOCK_STAT_TOKEN"]
        env["SELENIUM_EXECUTOR_JSON"] = json.dumps(browsers_config)
        env["PROFILE"] = json.dumps(last_released_profile)
        env["EXTENSIONS"] = json.dumps(dict(adblocks=self.get_adblock_extensions_path()))

        if self.Parameters.env_vars:
            for kv in self.Parameters.env_vars.strip().strip(';').split(';'):
                key, value = kv.split('=')
                env[key.strip()] = value.strip()

        cmd = [str(bin_res.path)]
        with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)

    def get_bin_res(self):
        if self.Parameters.binary:
            bin_res = sdk2.ResourceData(self.Parameters.binary)
        else:
            resource_class = AntiadblockDetectCheckerBin
            binary_id = sdk2.Resource.find(
                resource_type=resource_class,
                state=ctr.State.READY
            ).limit(1).first()
            bin_res = sdk2.ResourceData(binary_id)
        return bin_res
