# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.projects.common import binary_task
import sandbox.common.types.resource as ctr

from datetime import datetime


class RetryableException(Exception):
    pass


class AntiadblockNetworkChangesDetectorResult(sdk2.Resource):
    """
    Result of ANTIADBLOCK_NETWORK_CHANGES_DETECTOR task
    """
    auto_backup = True
    group = 'ANTIADBLOCK'


class AntiadblockNetworkChangesDetectorTask(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """Task that gets all ip networks and detects changes over time"""

    name = 'ANTIADBLOCK_NETWORK_CHANGES_DETECTOR'

    class Parameters(sdk2.Task.Parameters):
        network_list_url = sdk2.parameters.String('Network list site url', default_value='https://racktables.yandex.net/export/expand-fw-macro.php?macro=_NAT64COMMONNETS_')
        juggler_host = sdk2.parameters.String('Juggler host', default_value='antiadb_infra')
        juggler_service = sdk2.parameters.String('Juggler service', default_value='antiadb_network_changes')

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        import requests

        from retry import retry

        from antiadblock.tasks.tools.juggler import send_event

        @retry(RetryableException, tries=4, delay=1, backoff=2)
        def get_ips():
            response = requests.get(self.Parameters.network_list_url)
            if response.status_code in (502, 503, 504):
                raise RetryableException()
            response.raise_for_status()
            return sorted(item for item in response.content.split(b'\n') if item)

        # Get previous launch result if any
        prev_result = None
        prev_result_resource = sdk2.Resource.find(resource_type=AntiadblockNetworkChangesDetectorResult, state=ctr.State.READY).first()
        if prev_result_resource is not None:
            prev_result_resource_data = sdk2.ResourceData(prev_result_resource)
            prev_result = prev_result_resource_data.path.read_bytes().split(b',')

        # Load ips from https://racktables.yandex.net/export/expand-fw-macro.php?macro=_NAT64COMMONNETS_
        result = get_ips()

        # Save result as a sandbox resource
        result_resource = AntiadblockNetworkChangesDetectorResult(self, 'All networks', 'all-networks-{}'.format(datetime.now().strftime('%Y-%m-%d')))
        result_resource.path.write_bytes(b','.join(result))

        # Compare result with previous and if not equal lighten up juggler alert
        status = 'OK'
        description = 'Нет изменений в списке подсетей'
        if prev_result is not None and result != prev_result:
            status = 'CRIT'
            description = 'Произошли изменения в списке подсетей\n'
            description += 'Список: {}\n'.format(self.Parameters.network_list_url)
            result_set = set(result)
            prev_result_set = set(prev_result)
            if result_set - prev_result_set:
                description += 'Добавлены: {}\n'.format(' '.join(item.decode() for item in result_set - prev_result_set))
            if prev_result_set - result_set:
                description += 'Удалены: {}\n'.format(' '.join(item.decode() for item in prev_result_set - result_set))

        send_event(self.Parameters.juggler_host, self.Parameters.juggler_service, status, description=description)
