# -*- coding: utf8 -*-
import os
from datetime import datetime as dt, timedelta

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.antiadblock.run_bin import AntiadblockBinResource
from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


class AntiadblockBypassUidsTool(AntiadblockBinResource):
    """
    ANTIADBLOCK_BYPASS_UIDS_TOOL
    antiadblock/tasks/update_bypass/bin
    """
    pass


class AntiadblockBypassUidsModel(AntiadblockBinResource):
    """
    ANTIADBLOCK_BYPASS_UIDS_MODEL
    """
    pass


class AntiadblockBypassUidsDesktop(AntiadblockBinResource):
    """
    ANTIADBLOCK_BYPASS_UIDS_DESKTOP
    """
    pass


class AntiadblockBypassUidsMobile(AntiadblockBinResource):
    """
    ANTIADBLOCK_BYPASS_UIDS_MOBILE
    """
    pass


class BypassUidsTasksParameters(sdk2.Task.Parameters):
    binary_id = sdk2.parameters.LastReleasedResource(
        'Ресурс ADBLOCK_BYPASS_UIDS_TOOL, если нужно использовать не последний загруженный',
        resource_type=AntiadblockBypassUidsTool,
        state=(ctr.State.READY,),
        required=False
    )
    logs_range_start = sdk2.parameters.String(
        'Начало периода, логи за который нужно использовать (Y-m-d)',
        required=False
    )
    logs_range_end = sdk2.parameters.String(
        'Конец периода, логи за который нужно использовать (Y-m-d)',
        required=False
    )
    days_count = sdk2.parameters.Integer(
        'Количество предыдущих дней, логи за которые нужно использовать (используется, если не определен хотя бы один из logs_range_*)',
        required=False
    )
    sample_percent = sdk2.parameters.Integer(
        'Использовать для работы указанный процент от имеющихся данных [1, 100]',
        required=True,
        default=100,
    )
    use_prepared_data = sdk2.parameters.Bool(
        'Использовать приготовленные ранее данные',
        required=True,
        default=False,
    )


class AntiadblockTeachBypassUidsModel(sdk2.Task):
    """
    loads previously prepared training data & prepares catboost model
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        cores = 20
        disk_space = 100 * 2 ** 10  #: Minimum disk space in MiB
        ram = 100 * 2 ** 10  #: Minimum RAM size in MiB

    class Parameters(BypassUidsTasksParameters):
        pass

    def on_execute(self):
        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['SB_TOKEN'] = yav_secrets['ANTIADBLOCK_SANDBOX_TOKEN']
        cmd = [
            str(sdk2.ResourceData(self.Parameters.binary_id).path),
            '--logs_range', self.Parameters.logs_range_start, self.Parameters.logs_range_end,
            'model', '--train', '--upload',
            '--sample_percent', str(self.Parameters.sample_percent)
        ]
        with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)


class AntiadblockUploadBypassUidsModel(sdk2.Task):

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(BypassUidsTasksParameters):
        pass

    def _get_parameters(self):
        binary_id = self.Parameters.binary_id
        if not binary_id:
            binary_id = sdk2.Resource.find(
                resource_type=AntiadblockBypassUidsTool,
                state=ctr.State.READY
            ).limit(1).first()

        logs_range = (self.Parameters.logs_range_start, self.Parameters.logs_range_end)
        if not all(logs_range):
            if not getattr(self.Parameters, 'stream', False):
                days_count = self.Parameters.days_count or 3
                end = dt.now().replace(hour=0, minute=0, second=0, microsecond=0) - timedelta(days=1)
                start = end - timedelta(days=days_count-1)
                logs_range = list(map(lambda t: dt.strftime(t, '%Y-%m-%d'), [start, end]))
            else:
                end = dt.now().replace(minute=0, second=0, microsecond=0) + timedelta(hours=1)
                start = end - timedelta(hours=int(self.Parameters.hours))
                logs_range = list(map(lambda t: dt.strftime(t, '%Y-%m-%dT%H:%M:00'), [start, end]))

        sample_percent = self.Parameters.sample_percent
        use_prepared_data = self.Parameters.use_prepared_data
        return binary_id, logs_range, sample_percent, use_prepared_data

    def on_execute(self):
        binary_id, logs_range, sample_percent, use_prepared_data = self._get_parameters()
        bin_res = sdk2.ResourceData(binary_id)

        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['SB_TOKEN'] = yav_secrets['ANTIADBLOCK_SANDBOX_TOKEN']

        if not use_prepared_data:
            cmd = [
                str(bin_res.path),
                '--logs_range', logs_range[0], logs_range[1],
                'dataset', '--train',
            ]
            with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
                sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)

        child = AntiadblockTeachBypassUidsModel(
            self,
            description="Child of test task {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
        )
        child.Parameters.binary_id = binary_id
        child.Parameters.logs_range_start, child.Parameters.logs_range_end = logs_range
        child.Parameters.sample_percent = sample_percent
        child.Parameters.kill_timeout = 8 * 60 * 60
        child.save()
        child.enqueue()


class AntiadblockUploadBypassUids(AntiadblockUploadBypassUidsModel):

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        cores = 18
        ram = 18 * 2 ** 10
        disk_space = 100 * 2 ** 10  #: Minimum disk space in MiB

    class Parameters(BypassUidsTasksParameters):
        upload = sdk2.parameters.Bool(
            'Загрузить уиды как ресурсы в Sandbox',
            required=True,
            default=False,
        )
        stream = sdk2.parameters.Bool(
            'Использовать стримовые логи для создания датасета',
            required=True,
            default=False,
        )
        hours = sdk2.parameters.Integer(
            'Количество часов, за которое нужно использовать стримовые логи',
            required=True,
            default=5,
        )

    def on_execute(self):
        binary_id, logs_range, sample_percent, use_prepared_data = self._get_parameters()
        bin_res = sdk2.ResourceData(binary_id)

        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['SB_TOKEN'] = yav_secrets['ANTIADBLOCK_SANDBOX_TOKEN']

        if not use_prepared_data:
            cmd = [str(bin_res.path)]
            if self.Parameters.stream:
                cmd.append('--stream')
            cmd += [
                '--logs_range', logs_range[0], logs_range[1],
                'dataset', '--predict',
            ]
            with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
                sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)

        cmd = [str(bin_res.path)]
        if self.Parameters.stream:
            cmd.append('--stream')
        cmd += [
            '--logs_range', logs_range[0], logs_range[1],
            'predict',
            '--sample_percent', str(sample_percent)
        ]
        if self.Parameters.upload:
            cmd.append('--upload')
        with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)


class AntiadblockUpdateNginxUids(sdk2.Task):
    """
    update nginx uids in table at hahn
    """
    name = "ANTIADBLOCK_UPDATE_NGINX_UIDS"

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        cores = 1

    class Parameters(sdk2.Task.Parameters):
        binary_id = sdk2.parameters.LastReleasedResource(
            'Ресурс ADBLOCK_BYPASS_UIDS_TOOL, если нужно использовать не последний загруженный',
            resource_type=AntiadblockBypassUidsTool,
            state=(ctr.State.READY,),
            required=False
        )
        hours = sdk2.parameters.Integer(
            'Количество часов, которое считаем уиды актуальными',
            default=24,
            required=True,
        )

    def on_execute(self):
        binary_id = self.Parameters.binary_id
        if not binary_id:
            binary_id = sdk2.Resource.find(
                resource_type=AntiadblockBypassUidsTool,
                state=ctr.State.READY
            ).limit(1).first()
        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['SB_TOKEN'] = yav_secrets['ANTIADBLOCK_SANDBOX_TOKEN']
        uid_from_dt = dt.now() - timedelta(hours=int(self.Parameters.hours))
        cmd = [
            str(sdk2.ResourceData(binary_id).path),
            'update_nginx_uids',
            '--uid_from_dt', uid_from_dt.strftime('%Y-%m-%dT%H:%M:00')
        ]
        with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)


class AntiadblockMakeBypasUids(sdk2.Task):
    """
    make bypass uids form predict data and stream logs
    """
    name = "ANTIADBLOCK_MAKE_BYPASS_UIDS"

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        cores = 18
        ram = 18 * 1024
        disk_space = 100 * 1024

    class Parameters(sdk2.Task.Parameters):
        binary_id = sdk2.parameters.LastReleasedResource(
            'Ресурс ADBLOCK_BYPASS_UIDS_TOOL, если нужно использовать не последний загруженный',
            resource_type=AntiadblockBypassUidsTool,
            state=(ctr.State.READY,),
            required=False
        )
        use_predict_uids = sdk2.parameters.Bool(
            'Использовать выборку, полученную с помощью модели',
            required=True,
            default=False,
        )
        use_stream_predicts = sdk2.parameters.Bool(
            'Дополнительно использовать выборку, полученную из стримовых логов',
            required=True,
            default=False,
        )
        use_not_blocked_uids = sdk2.parameters.Bool(
            'Использовать выборку not_blocked уидов, полученных из стримовых логов',
            required=True,
            default=False,
        )
        remove_blocked_uids = sdk2.parameters.Bool(
            'Исключать из выборки blocked уиды, полученные из стримовых логов',
            required=True,
            default=False,
        )
        remove_uids_without_cycada = sdk2.parameters.Bool(
            'Исключать из выборки уиды, запросы которых без куки cycada',
            required=True,
            default=False,
        )
        ratio = sdk2.parameters.Float(
            'Коэффициент для  определения статуса уида, отношение событий not_blocked (blocked) к общему числу bamboozled событий',
            default=0.9,
            required=True,
        )
        min_cnt = sdk2.parameters.Integer(
            'Минимальное количество bamboozled событий для учитывания уида',
            default=50,
            required=True,
        )
        upload = sdk2.parameters.Bool(
            'Загрузить уиды как ресурсы в Sandbox',
            required=True,
            default=False,
        )

    def on_execute(self):
        binary_id = self.Parameters.binary_id
        if not binary_id:
            binary_id = sdk2.Resource.find(
                resource_type=AntiadblockBypassUidsTool,
                state=ctr.State.READY
            ).limit(1).first()
        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['SB_TOKEN'] = yav_secrets['ANTIADBLOCK_SANDBOX_TOKEN']
        cmd = [
            str(sdk2.ResourceData(binary_id).path),
            'make_bypass_uids',
            '--ratio', str(self.Parameters.ratio),
            '--min_cnt', str(self.Parameters.min_cnt),
        ]
        if self.Parameters.use_predict_uids:
            cmd.append('--use_predict_uids')
        if self.Parameters.use_stream_predicts:
            cmd.append('--use_stream_predicts')
        if self.Parameters.use_not_blocked_uids:
            cmd.append('--use_not_blocked_uids')
        if self.Parameters.remove_blocked_uids:
            cmd.append('--remove_blocked_uids')
        if self.Parameters.remove_uids_without_cycada:
            cmd.append('--remove_uids_without_cycada')
        if self.Parameters.upload:
            cmd.append('--upload')
        with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)
