import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

from sandbox.projects.antiadblock.utils.yt_helpers import get_available_yt_clusters
from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


class AntiadblockMoneyByServiceIdBin(sdk2.Resource):
    """
    antiadblock/tasks/money_by_service_id binary
    """
    releasable = True
    any_arch = True
    auto_backup = True
    group = 'ANTIADBLOCK'
    releasers = ['ANTIADBLOCK']
    release_subscribers = ["ANTIADBLOCK"]


class AntiadblockMoneyByServiceId(sdk2.Task):
    """ antiadblock/tasks/money_by_service_id binary from SB resource"""

    class Requirements(sdk2.Requirements):
        cores = 1
        client_tags = ctc.Tag.LINUX_XENIAL
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='ANTIADBLOCK_MONEY_BY_SERVICE_ID', weight=1, capacity=2)
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH, ctt.Status.Group.WAIT)
        )

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        daily_logs = sdk2.parameters.Bool('Use daily logs', default=False)
        env_vars = sdk2.parameters.String('Optional environment variables (expecting "var1=val1;var2=val2")', required=False)

    def on_execute(self):
        binary_id = sdk2.Resource.find(
            resource_type=AntiadblockMoneyByServiceIdBin,
            state=ctr.State.READY,
        ).limit(1).first()
        bin_res = sdk2.ResourceData(binary_id)
        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['STAT_TOKEN'] = yav_secrets["ANTIADBLOCK_STAT_TOKEN"]
        env['SOLOMON_TOKEN'] = yav_secrets['ANTIADBLOCK_SOLOMON_TOKEN']

        # get available yt clusters
        env['AVAILABLE_YT_CLUSTERS'] = get_available_yt_clusters()
        logging.info('AVAILABLE YT CLUSTERS: {}'.format(env['AVAILABLE_YT_CLUSTERS']))

        if self.Parameters.env_vars:
            for kv in self.Parameters.env_vars.strip().strip(';').split(';'):
                key, value = kv.split('=')
                env[key.strip()] = value.strip()

        cmd = [str(bin_res.path)]
        if self.Parameters.daily_logs:
            cmd.append('--daily_logs')
        with sdk2.helpers.ProcessLog(self, logger="run_task_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)
