import os
import time
import uuid
import shutil
import urllib2
import tarfile
import logging
import datetime as dt

from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess as sp
from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated
from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.qa.server_interface import LocalServerModuleInterface

from sandbox.projects.antiadblock.qa.utils.constants import ANTIADBLOCK_COOKIEMATCHER_CRYPT_KEYS, CRYPROX_TVM2_SECRET, DiffType
# from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


class EngineModule(ModuleBase, LocalServerModuleInterface):

    def __init__(self, adapter, stub_dir, test_type, configs_stub_port, profile_clock_type):
        ModuleBase.__init__(self, adapter)
        self.instance_id = str(uuid.uuid4())
        self.engine_path = self.adapter.get_engine_executable_path()
        self._ports = {'nginx': 80}
        self._engine_process = None
        self.configs_stub_port = configs_stub_port
        self.profile_clock_type = profile_clock_type

        # LATENCY, CAPACITY, END2END, PROFILING
        self.test_type = test_type

        # ensure required dirs
        self.task_dir = stub_dir if stub_dir else str(self.task.path())
        self.work_dir = os.path.join(self.task_dir, self.instance_id)
        self.logs_dir = os.path.join(get_logs_folder(), self.instance_id)
        os.mkdir(self.work_dir)
        os.mkdir(self.logs_dir)

        # cache_daemon
        self._cache_daemon_services = {
            'cryprox': ['cryprox'],
        }
        if test_type != DiffType.END2END:
            self._cache_daemon_services['accel_redirect'] = ['accel_redirect']
        n_threads = int(self.adapter.get_nginx_worker_count()) * 2
        self._cache_daemon = CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=self.adapter.get_cache_daemon_executable_path(),
            dump_path=self.adapter.get_cache_daemon_stub_path(),
            data_dir=os.path.join(self.work_dir, 'cache_daemon_data'),
            log_subdir=os.path.join(self.logs_dir, 'cache_daemon_logs'),
            start_on_creation=False, services=self._cache_daemon_services,
            key_header=self.adapter.get_cache_daemon_key_headers(),
            n_threads=n_threads,
            max_queue_size=500*n_threads,
        )

        self._ports.update(self._cache_daemon.get_ports_by_tag())
        self._prepare_engine()
        logging.info("Using ports: {}".format(self._ports))
        logging.info("Key header: {}".format(self.adapter.get_cache_daemon_key_headers()))

    def get_process(self):
        return self._engine_process

    def get_port(self):
        return self._ports['nginx']

    def wait_ping(self, port, tag):
        timeout_seconds = 300
        start_dt = dt.datetime.now()
        while dt.datetime.now() - start_dt < dt.timedelta(seconds=timeout_seconds):
            # noinspection PyBroadException
            try:
                request = urllib2.Request("http://localhost:{}/ping".format(port))
                urllib2.urlopen(request, timeout=10)
                logging.info('{} started!'.format(tag))
                return True
            except Exception:
                time.sleep(5)
        raise Exception('{} did not start in {} seconds!'.format(tag, timeout_seconds))

    def make_cryprox_env(self):
        env = os.environ.copy()
        env["ENV_TYPE"] = "load_testing"
        env["LOGGING_LEVEL"] = "DEBUG"
        env["REQUEST_TIMEOUT"] = "60"
        # always get configs form stub
        env["CONFIGSAPI_URL"] = "http://localhost:{}/".format(self.configs_stub_port)
        env["CRYPROX_TVM2_CLIENT_ID"] = "2001021"
        env["CRYPROX_TVM2_SECRET"] = sdk2.Vault.data(CRYPROX_TVM2_SECRET)  # sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()[CRYPROX_TVM2_SECRET]
        env["ADMIN_TVM2_CLIENT_ID"] = "2000629"
        env["REDIS_HOSTS"] = ""
        env["WORKERS_COUNT"] = "{}".format(self.adapter.get_cryprox_worker_count())
        env["STUB_SERVER_PORT"] = str(self._ports['cryprox'])
        if self.test_type == DiffType.END2END:
            env["END2END_TESTING"] = "1"
        elif self.test_type == DiffType.PROFILING:
            env["PROFILING"] = "1"
            env["YAPPI_CLOCK_TYPE"] = self.profile_clock_type
        return env

    def make_nginx_env(self):
        env = os.environ.copy()
        env["WORKERS_COUNT"] = "{}".format(self.adapter.get_nginx_worker_count())
        env["CRYPROX"] = "[::1]:8081"
        if self.test_type != DiffType.END2END:
            env["ACCELREDIRECT_TARGET"] = "proxy_pass http://localhost:{}/$is_args$args;".format(self._ports['accel_redirect'])
        else:
            env["ACCELREDIRECT_TARGET"] = ""
        env["EXTERNAL_RESOLVER"] = "[2a02:6b8:0:3400::5005]"
        env["UNICERT_SSL"] = ""
        return env

    def clean(self):
        logging.info('Remove cryprox package')
        for _dir in ('etc', 'usr', 'bin'):
            shutil.rmtree(os.path.join(os.getcwd(), _dir))

        logging.info('Remove nginx confs')
        for _cfg in ('static-mon.conf', 'cookiematcher.conf', 'cryprox.conf', 'naydex.conf'):
            os.remove(os.path.join('/etc/nginx/sites-enabled', _cfg))
        shutil.rmtree('/etc/nginx/sites-available')
        shutil.rmtree('/etc/nginx/conf.d')
        os.remove('/usr/bin/slb')
        for _file in ('nginx.conf.template', 'worker.lua', 'ssl', 'make_conf.sh', 'jwt_partner.lua', 'launch.sh'):
            os.remove(os.path.join('/etc/nginx', _file))

        logging.info('Remove cryprox')
        shutil.rmtree('/bin/cryprox_run')
        shutil.rmtree('/etc/cookiematcher')
        shutil.rmtree('/perm')
        if os.path.exists('/usr/bin/update_resources.py'):
            os.remove('/usr/bin/update_resources.py')
        elif os.path.exists('/usr/bin/update_bypass_uids.py'):
            os.remove('/usr/bin/update_bypass_uids.py')

        logging.info('Remove engine logs')
        for log in ('cryprox', 'nginx'):
            shutil.rmtree(os.path.join('/logs', log))

        logging.info('Remove work_dir: {}'.format(self.work_dir))
        shutil.rmtree(self.work_dir)

    def _prepare_engine(self):
        with sdk2.helpers.ProcessLog(self.task, logger="prepare_engine_log") as log:
            logging.info("Extract cryprox package")
            with tarfile.open(self.adapter.get_cryprox_package_path()) as tar:
                tar.extractall()
            prepare_script_path = os.path.join(os.getcwd(), "usr/bin/init_instance_for_sandbox.sh")
            logging.info("Run init_instance_for_sandbox.sh")
            cmd = [prepare_script_path]
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=self.make_cryprox_env())

            prepare_script_path = os.path.join(os.getcwd(), "usr/bin/make_nginx_conf.sh")
            logging.info("Run make_nginx_conf.sh")
            cmd = [prepare_script_path]
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=self.make_nginx_env())

            cookiematcher_crypt_keys = sdk2.Vault.data(ANTIADBLOCK_COOKIEMATCHER_CRYPT_KEYS)  # sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()[ANTIADBLOCK_COOKIEMATCHER_CRYPT_KEYS]
            with open("/etc/cookiematcher/cookiematcher_crypt_keys.txt", mode="w") as fout:
                fout.write(cookiematcher_crypt_keys)

    def __enter__(self):
        self._cache_daemon.__enter__()

        self._engine_log = ProcessLog(logger='engine_log')
        self._engine_log.__enter__()
        self._engine_process = sp.Popen([self.engine_path],
                                        stdout=self._engine_log.stdout, stderr=self._engine_log.stderr,
                                        cwd=self.work_dir)
        ProcessRegistry.register(self._engine_process.pid, [self.engine_path])
        # wait nginx
        self.wait_ping(self.get_port(), "Nginx")
        # wait cryprox service app
        self.wait_ping(8080, "Cryprox")
        return self

    def __exit__(self, *args):
        logging.info('Exiting engine module!')
        self._cache_daemon.__exit__(*args)
        time.sleep(15)  # sleep to flush all log records
        with sdk2.helpers.ProcessLog(self.task, logger="stop_nginx_log") as log:
            cmd = [self.engine_path, '-s', 'stop']
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr)

        self._engine_log.__exit__(*args)

    @property
    def task(self):
        return self.adapter.task_instance
