import os
import time
import logging
from collections import defaultdict

from sandbox import sdk2
from sandbox.sdk2.os import enable_turbo_boost, disable_turbo_boost

from sandbox.projects.yabs.qa.ammo_module.dplan.adapters.sandbox import AmmoDplanModuleSandboxAdapter
from sandbox.projects.yabs.qa.dumper_module.adapters.sandbox import DumperModuleSandboxAdapter

from sandbox.projects.antiadblock.qa.modules.configs_stub_module.adapters.sandbox import ConfigsStubSandboxAdapter
from sandbox.projects.antiadblock.qa.modules.engine_module.adapters.sandbox import EngineSandboxAdapter
from sandbox.projects.antiadblock.qa.tasks.performance_shoot_diff import AntiadblockPerformanceShootsDiff, DolbiloModuleTmpfsSandboxAdapter, make_prefix
from sandbox.projects.antiadblock.qa.tasks.parameters import BaseDiffShootParameters
from sandbox.projects.antiadblock.qa.utils.constants import DiffType, PERCENTILES


logger = logging.getLogger("run_task_log")


class AntiadblockDetectCapacityShoot(AntiadblockPerformanceShootsDiff):
    name = 'ANTIADBLOCK_DETECT_CAPACITY_SHOOT'

    class Parameters(BaseDiffShootParameters):
        description = "Antiadblock diff detect capacity performance shoots task"

        with sdk2.parameters.Group('General settings') as general_settings:
            error_rate_thr = sdk2.parameters.Float('Error rate threshold', default_value=0.003, required=True)
            rps_diff_thr = sdk2.parameters.Float('RPS diff threshold', default_value=0.2, required=True)
            count_shoot_sessions = sdk2.parameters.Integer('Count shoot sessions', default_value=5, required=True)
            push_to_stat = sdk2.parameters.Bool('Push perf stat to report', default=False, required=True)
            disable_turbo_boost = sdk2.parameters.Bool('Disable Turbo Boost', default=False, required=True)

        with sdk2.parameters.Output:
            result_stats = sdk2.parameters.JSON('Output processed stats', default={})
            has_diff = sdk2.parameters.Bool('Shoots has diff', default=False)

    def run_pipeline(self, prefix='0'):
        self.dumper_module = DumperModuleSandboxAdapter(self.Parameters.dumper_parameters, self).create_module()
        self.ammo_module = AmmoDplanModuleSandboxAdapter(self.Parameters.ammo_parameters, self).create_module()
        self.shoot_module = DolbiloModuleTmpfsSandboxAdapter(self.Parameters.dolbilo_parameters, self).create_module()

        self.shoot_module.adapter.parameters.circular_session = True  # reuse bullets

        with self.engine_service as active_service:
            # warm up
            self.shoot_module.adapter.parameters.mode = 'plan'
            self.warm_up(active_service, logger)
            # main shoot
            self.shoot_module.adapter.parameters.mode = 'finger'  # shoot mode
            dump_path = self.shoot_module.shoot(active_service, self.ammo_module.get_dplan_path())

        rps = float(self.dumper_module.get_processed_stats(dump_path).get('rps', '0'))
        os.remove(dump_path)
        self.save_resources(dump_path=None, prefix=prefix)
        return rps

    def on_execute(self):
        if self.Parameters.disable_turbo_boost:
            disable_turbo_boost()
        stub_dir = str(self.ramdrive.path) if self.ramdrive else ''
        result_rps = defaultdict(list)
        self.configs_stub_module = ConfigsStubSandboxAdapter(self.Parameters.configs_stub_parameters, self).create_module()
        with self.configs_stub_module as active_stub:
            # wait while stub started
            time.sleep(180)
            for i, engine_parameters in enumerate([self.Parameters.first_engine_parameters, self.generate_second_engine_parameters()]):
                # reload configs
                self.reload_configs(engine_parameters.replaced_configs, active_stub.get_port(), logger)
                for session in range(int(self.Parameters.count_shoot_sessions)):
                    logger.info("Start session {}".format(session))
                    self.engine_service = EngineSandboxAdapter(engine_parameters, self).create_module(stub_dir, DiffType.CAPACITY, active_stub.get_port())
                    rps = self.run_pipeline(prefix=make_prefix(i, session))
                    result_rps['shoot {}'.format(i)].append(rps)
                    self.engine_service.clean()
                    # wait when all ports unbinding
                    time.sleep(180)

        self.make_diff(quantilies=PERCENTILES, rps=result_rps)
        self.Parameters.result_stats = {'rps': result_rps}
        # return Turbo Boost
        if self.Parameters.disable_turbo_boost:
            enable_turbo_boost()
