# coding=utf-8

import os
import time
import json
import shutil
import logging
from collections import defaultdict, OrderedDict
from urlparse import urlparse
from urllib import urlencode

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.client as ctc

from sandbox.projects.yabs.qa.ammo_module.dplan.adapters.sandbox import AmmoDplanModuleSandboxAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import DolbiloModuleSandboxAdapter

from sandbox.projects.antiadblock.qa.modules.configs_stub_module.adapters.sandbox import ConfigsStubSandboxAdapter
from sandbox.projects.antiadblock.qa.modules.engine_module.adapters.sandbox import EngineSandboxAdapter
from sandbox.projects.antiadblock.qa.resource_types import ANTIADBLOCK_FUNC_DIFF, ANTIADBLOCK_CRYPROX_REGEX_STAT
from sandbox.projects.antiadblock.qa.tasks.performance_shoot_diff import AntiadblockPerformanceShootsDiff, parse_nginx_access_log
from sandbox.projects.antiadblock.qa.tasks.parameters import BaseFuncDiffShootParameters
from sandbox.projects.antiadblock.qa.utils.constants import DiffType, FILE_CRYPROX_END2END_LOG_TMPL, FILE_NGINX_ACCESS_LOG_TMPL, \
    FILE_CRYPROX_LOG_TMPL, FILE_CRYPROX_REGEX_STAT_TMPL


logger = logging.getLogger("run_task_log")


def make_prefix(i):
    return '{}'.format(i)


def get_content_from_body(body):
    result = body.split('\r\n')
    if len(result) > 1:
        return body.split('\r\n')[-2]
    return ""


class ResponseRecord(object):
    KEYS = ("service_id", "status_code", "request_tags", "request_url", "body")

    def __init__(self, data):
        self._request_id = data["request_id"]
        for key in self.KEYS:
            if key == "request_url":
                url = urlparse(data[key])
                query = {}
                if url.query:
                    query = OrderedDict(sorted([p.split('=', 1) for p in url.query.split('&') if '=' in p]))
                setattr(self, key, url._replace(scheme='http', netloc='localhost', query=urlencode(query)))
            elif key == "body":
                setattr(self, key, get_content_from_body(data[key]))
            elif key == "request_tags":
                setattr(self, key, sorted(data[key]))
            else:
                setattr(self, key, data[key])

    def __getitem__(self, item):
        return getattr(self, item)

    def get_diff_keys(self, other):
        return [key for key in self.KEYS if self[key] != other[key]]

    def to_dict(self, with_body=False):
        return {key: self[key] for key in self.KEYS if key != "body" or with_body}

    @property
    def request_id(self):
        return self._request_id


def parse_end2end_log(log_path, logger):
    logger.info("Parse end2end logs: {}".format(log_path))
    results = {}
    count, err_count = 0, 0
    with open(log_path) as fin:
        for line in fin:
            count += 1
            try:
                record = ResponseRecord(json.loads(line))
                results[record.request_id] = record
            except Exception as ex:
                err_count += 1
                logger.debug("Can't parse record: {}\nexception: {} ".format(line.strip()[:1000], str(ex)[:1000]))
    msg = "Count rows: {}, parse errors: {}".format(count, err_count)
    logger.info(msg)
    return results, msg


def parse_regex_stat(cryprox_log_file, file_out, logger):
    logger.info("Parse regex stat from cryprox log: {}".format(cryprox_log_file))
    results = defaultdict(lambda: defaultdict(set))

    with open(cryprox_log_file) as fin:
        for line in fin:
            record = json.loads(line.strip())
            service_id = record.get("service_id", "")
            if service_id and "regex" in record:
                regex = record["regex"]
                url = record["url"]
                results[service_id][regex].add(url)
    with open(file_out, 'w') as fout:
        for service_id, regex_stat in sorted(results.items()):
            fout.write(service_id)
            fout.write('\n')
            for regex, urls in sorted(regex_stat.items()):
                fout.write(regex)
                fout.write('\n')
                fout.write("\n".join(sorted(urls)))
                fout.write('\n\n')
            fout.write('\n')


class AntiadblockFuncShootsDiff(AntiadblockPerformanceShootsDiff):
    name = 'ANTIADBLOCK_FUNC_SHOOTS_DIFF'

    class Requirements(sdk2.Task.Requirements):
        privileged = True  # for root
        cores = 8
        disk_space = 80 * 1024  # GB
        ram = 40 * 1024  # GB + 20 GB for cached stub in tmpfs
        client_tags = ctc.Tag.LINUX_XENIAL

    class Context(sdk2.Task.Context):
        has_diff = True

    class Parameters(BaseFuncDiffShootParameters):
        description = 'Antiadblock functional shoot diff task'

        with sdk2.parameters.Group('General settings') as general_settings:
            error_rate_thr = sdk2.parameters.Float('Error rate threshold', default_value=0.003, required=True)

        with sdk2.parameters.Output:
            has_diff = sdk2.parameters.Bool('Shoots has diff', default=False)

    def make_func_diff(self):
        end2end_logs = [None, None]
        func_diff = {'not_exist': {}, 'diff': defaultdict(dict)}
        count_diff = 0
        errors = []
        engine_logs_path = getattr(self, 'engine_logs_path', '')
        report = ['<b>General info</b>']
        for _ind in (0, 1):
            shoot = 'shoot {}'.format(_ind)
            nginx_access_stat = parse_nginx_access_log(os.path.join(engine_logs_path, FILE_NGINX_ACCESS_LOG_TMPL.format(make_prefix(_ind))),
                                                       None, logger)
            # errors
            for error in ('count_server_errors', 'count_client_errors'):
                if nginx_access_stat['all'][error] > float(self.Parameters.error_rate_thr) * nginx_access_stat['all']['count_req']:
                    message = 'Too many {} in our nginx access log ({}): {}'.format(error, shoot, nginx_access_stat['all'][error])
                    errors.append(message)

            end2end_logs[_ind], msg = parse_end2end_log(os.path.join(engine_logs_path, FILE_CRYPROX_END2END_LOG_TMPL.format(make_prefix(_ind))), logger)
            report.extend(['Parse logs for {}'.format(shoot), msg])
            regex_stat_filename = FILE_CRYPROX_REGEX_STAT_TMPL.format(make_prefix(_ind))
            parse_regex_stat(os.path.join(engine_logs_path, FILE_CRYPROX_LOG_TMPL.format(make_prefix(_ind))), regex_stat_filename, logger)
            regex_stat_resource = ANTIADBLOCK_CRYPROX_REGEX_STAT(self, 'Cryprox regex stat for shoot {}'.format(make_prefix(_ind)), regex_stat_filename)
            sdk2.ResourceData(regex_stat_resource).ready()

        count_response_on_type = {'with_diff': defaultdict(int), 'without_diff': defaultdict(int)}

        for request_id in end2end_logs[0].keys():
            if request_id not in end2end_logs[1]:
                count_diff += 1
                func_diff['not_exist'][request_id] = end2end_logs[0][request_id].to_dict()
                continue
            diff_keys = end2end_logs[0][request_id].get_diff_keys(end2end_logs[1][request_id])
            if diff_keys:
                count_diff += 1
                func_diff['diff'][request_id]['keys'] = diff_keys
                with_body = "body" in diff_keys
                func_diff['diff'][request_id][0] = end2end_logs[0][request_id].to_dict(with_body)
                func_diff['diff'][request_id][1] = end2end_logs[1][request_id].to_dict(with_body)
                count_response_on_type['with_diff'][', '.join([end2end_logs[0][request_id]['service_id']] + end2end_logs[0][request_id]['request_tags'])] += 1
            else:
                count_response_on_type['without_diff'][', '.join([end2end_logs[0][request_id]['service_id']] + end2end_logs[0][request_id]['request_tags'])] += 1

        report.append('Count diff records: {}'.format(count_diff))
        report.append('Not exists: {}'.format(len(func_diff['not_exist'])))
        report.append('Real diff: {}'.format(len(func_diff['diff'])))
        for _type in ('with_diff', 'without_diff'):
            report.append('<b>Count responses {} on type</b>'.format(_type))
            for k, v in sorted(count_response_on_type[_type].items()):
                report.append('{}: {}'.format(k, v))

        filename = 'shoots_diff.log'

        with open(filename, mode="w") as fin:
            json.dump(count_response_on_type, fin)
            fin.write('\n\n')
            json.dump(func_diff, fin)
        diff_resource = ANTIADBLOCK_FUNC_DIFF(self, 'Func shoots diff', filename)
        sdk2.ResourceData(diff_resource).ready()

        self.Context.has_diff = count_diff > 0
        self.Context.report = '<br />'.join(report)
        self.Context.save()

        if errors:
            raise TaskFailure('\n'.join(errors))

        self.Parameters.has_diff = self.Context.has_diff

    def save_engine_logs(self, prefix='0'):
        super(AntiadblockFuncShootsDiff, self).save_engine_logs(prefix)
        engine_logs_path = getattr(self, 'engine_logs_path', '')
        cryprox_end2end_log_path = os.path.join(engine_logs_path, FILE_CRYPROX_END2END_LOG_TMPL.format(prefix))
        shutil.copy('/logs/cryprox/end2end_testing.log', cryprox_end2end_log_path)
        # TODO: fix save resources
        # cryprox_end2end_log_resource = ANTIADBLOCK_CRYPROX_END2END_LOG(self, 'CRYPROX end2end logs for shoot {}'.format(prefix),
        #                                                                cryprox_end2end_log_path)
        # sdk2.ResourceData(cryprox_end2end_log_resource).ready()

    def run_pipeline(self, prefix='0'):
        self.ammo_module = AmmoDplanModuleSandboxAdapter(self.Parameters.ammo_parameters, self).create_module()
        self.shoot_module = DolbiloModuleSandboxAdapter(self.Parameters.dolbilo_parameters, self).create_module()

        with self.engine_service as active_service:
            self.shoot_module.shoot(active_service, self.ammo_module.get_dplan_path(), store_dump=False)

        self.save_resources(dump_path=None, prefix=prefix)

    @sdk2.report(title="Shoots diff")
    def report(self):
        return self.Context.report or "No report discovered in context"

    def on_execute(self):
        stub_dir = str(self.ramdrive.path) if self.ramdrive else ''
        self.configs_stub_module = ConfigsStubSandboxAdapter(self.Parameters.configs_stub_parameters, self).create_module()
        with self.configs_stub_module as active_stub:
            # wait while stub started
            time.sleep(180)
            for i, engine_parameters in enumerate([self.Parameters.first_engine_parameters, self.generate_second_engine_parameters()]):
                # reload configs
                self.reload_configs(engine_parameters.replaced_configs, active_stub.get_port(), logger)
                logger.info("Shoot {}".format(i))
                self.engine_service = EngineSandboxAdapter(engine_parameters, self).create_module(stub_dir, DiffType.END2END, active_stub.get_port())
                self.run_pipeline(prefix=make_prefix(i))
                self.engine_service.clean()
                # wait when all ports unbinding
                time.sleep(180)

        logger.info("Make results diff")
        self.make_func_diff()
