import os
import logging
import shutil

from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.sandboxsdk import environments

from sandbox.projects.yabs.qa.ammo_module.dplan.adapters.sandbox import AmmoDplanModuleSandboxAdapter
from sandbox.projects.yabs.qa.dolbilo_module.simple.adapters.sandbox import DolbiloModuleSandboxAdapter
from sandbox.projects.yabs.qa.dumper_module.adapters.sandbox import DumperModuleSandboxAdapter

from sandbox.projects.antiadblock.qa.modules.engine_module.adapters.sandbox import EngineSandboxAdapter
from sandbox.projects.antiadblock.qa.resource_types import ANTIADBLOCK_PERFORMANCE_DUMP, ANTIADBLOCK_SHOOT_RAW_STATS
from sandbox.projects.antiadblock.qa.tasks.parameters import BasePerformanceShootParameters
from sandbox.projects.antiadblock.qa.utils.constants import FILE_RAW_STATS_TMPL, FILE_NGINX_ACCESS_LOG_TMPL, FILE_NGINX_ERROR_LOG_TMPL, FILE_CRYPROX_LOG_TMPL

logger = logging.getLogger("run_task_log")


class AntiadblockOnePerformanceShoot(sdk2.Task):
    name = 'ANTIADBLOCK_ONE_PERFORMANCE_SHOOT'

    class Requirements(sdk2.Task.Requirements):
        privileged = True  # for root
        cores = 22
        disk_space = 80 * 1024  # GB
        ram = 40 * 1024  # GB + 20 GB for cached stub in tmpfs
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.INTEL_E5_2660V4 & (ctc.Tag.SAS | ctc.Tag.VLA | ctc.Tag.MAN)
        environments = [
            environments.PipEnvironment('python-statface-client', use_user=False),
        ]

    class Parameters(BasePerformanceShootParameters):
        description = "Antiadblock performance shoot task"

        with sdk2.parameters.Group('General settings') as general:
            rps = sdk2.parameters.Integer('RPS', default_value=100, required=True)
            shoot_time = sdk2.parameters.Integer('Shoot time in seconds',
                                                 default_value=60, required=True)

        with sdk2.parameters.Group('Warmup settings') as warmup_settings:
            warmup_rps = sdk2.parameters.Integer('Warmup RPS', default_value=50, required=True)
            warmup_time = sdk2.parameters.Integer('Warmup time in seconds', default_value=60, required=True)

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)
        self.Requirements.ramdrive = ctm.RamDrive(
            ctm.RamDriveType.TMPFS,
            20480,  # 20 GB
            os.path.join(str(self.path()), 'stub_data'),
        )

    def save_engine_logs(self, prefix='0'):
        engine_logs_path = getattr(self, 'engine_logs_path', None)
        if engine_logs_path is None:
            # create engine_logs_path
            engine_logs_path = os.path.join(get_logs_folder(), 'engine_logs')
            os.mkdir(engine_logs_path)
            setattr(self, 'engine_logs_path', engine_logs_path)

        nginx_access_log_path = os.path.join(engine_logs_path, FILE_NGINX_ACCESS_LOG_TMPL.format(prefix))
        shutil.copy('/logs/nginx/access.log', nginx_access_log_path)
        # TODO: fix save resources
        # access_log_resource = ANTIADBLOCK_NGINX_ACCESS_LOG(self, 'NGINX access logs for shoot {}'.format(prefix),
        #                                                    nginx_access_log_path)
        # sdk2.ResourceData(access_log_resource).ready()

        nginx_error_log_path = os.path.join(engine_logs_path, FILE_NGINX_ERROR_LOG_TMPL.format(prefix))
        shutil.copy('/logs/nginx/error.log', nginx_error_log_path)
        # error_log_resource = ANTIADBLOCK_NGINX_ERROR_LOG(self, 'NGINX error logs for shoot {}'.format(prefix),
        #                                                  nginx_error_log_path)
        # sdk2.ResourceData(error_log_resource).ready()

        cryprox_log_path = os.path.join(engine_logs_path, FILE_CRYPROX_LOG_TMPL.format(prefix))
        shutil.copy('/logs/cryprox/cryprox.log', cryprox_log_path)
        # cryprox_log_resource = ANTIADBLOCK_CRYPROX_LOG(self, 'CRYPROX logs for shoot {}'.format(prefix),
        #                                                cryprox_log_path)
        # sdk2.ResourceData(cryprox_log_resource).ready()
        logging.info("Success save engine logs")
        logging.info(os.listdir(engine_logs_path))

    @staticmethod
    def clean_engine_logs():
        for log in ('/logs/nginx/access.log', '/logs/nginx/error.log', '/logs/cryprox/cryprox.log'):
            open(log, 'w').close()

    def create_all_modules(self):
        self.dumper_module = DumperModuleSandboxAdapter(self.Parameters.dumper_parameters, self).create_module()
        self.ammo_module = AmmoDplanModuleSandboxAdapter(self.Parameters.ammo_parameters, self).create_module()
        self.shoot_module = DolbiloModuleSandboxAdapter(self.Parameters.dolbilo_parameters, self).create_module()
        stub_dir = str(self.ramdrive.path) if self.ramdrive else ''
        self.engine_service = EngineSandboxAdapter(self.Parameters.engine_parameters, self).create_module(stub_dir)

    def save_resources(self, dump_path=None, prefix='0'):
        result = {'processed_stats': {}}
        if dump_path is not None:
            result['processed_stats'] = self.dumper_module.get_processed_stats(dump_path, add_raw_stats=True)
            logging.info(result)
            perf_dump_resource = ANTIADBLOCK_PERFORMANCE_DUMP(self, 'Result resource for shoot  {}'.format(prefix), dump_path)
            sdk2.ResourceData(perf_dump_resource).ready()

            with open(FILE_RAW_STATS_TMPL.format(prefix), mode="w") as fin:
                fin.write(result["processed_stats"]["raw_stats"])
            raw_stats_resource = ANTIADBLOCK_SHOOT_RAW_STATS(self, 'Raw stats for shoot  {}'.format(prefix), FILE_RAW_STATS_TMPL.format(prefix))
            sdk2.ResourceData(raw_stats_resource).ready()
            del result["processed_stats"]["raw_stats"]

        self.save_engine_logs(prefix=prefix)
        return result

    def run_pipeline(self):
        self.create_all_modules()
        # setup module for shooting with fixed rps
        self.shoot_module.adapter.parameters.circular_session = True  # reuse bullets
        self.shoot_module.adapter.parameters.mode = 'plan'  # shoot mode

        with self.engine_service as active_service:
            self.warm_up(active_service, logger)
            # main shoot
            # fixed rps and shooting time
            self.shoot_module.adapter.parameters.mode_arg = ['--rps-schedule', 'const({}, {})'.format(self.Parameters.rps,
                                                                                                      self.Parameters.shoot_time)]

            dump_path = self.shoot_module.shoot(active_service, self.ammo_module.get_dplan_path())

        return self.save_resources(dump_path)

    def on_execute(self):
        result = self.run_pipeline()
        self.Parameters.result_stats = result["processed_stats"]

    def warm_up(self, active_service, logger=logging.getLogger("run_task_log")):
        logger.info('Start warm up session: {} rps, {} seconds'.format(self.Parameters.warmup_rps, self.Parameters.warmup_time))
        # fixed rps and shooting time
        self.shoot_module.adapter.parameters.mode_arg = ['--rps-schedule', 'const({}, {})'.format(self.Parameters.warmup_rps,
                                                                                                  self.Parameters.warmup_time)]

        self.shoot_module.shoot(active_service, self.ammo_module.get_dplan_path(), store_dump=False)
        self.clean_engine_logs()
        logger.info('End warm up session')
