# coding=utf-8

import os
import shutil
import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
import sandbox.common.types.client as ctc

from sandbox.projects.antiadblock.qa.modules.configs_stub_module.adapters.sandbox import ConfigsStubSandboxAdapter
from sandbox.projects.antiadblock.qa.modules.engine_module.adapters.sandbox import EngineSandboxAdapter
from sandbox.projects.antiadblock.qa.resource_types import ANTIADBLOCK_CALLGRIND_STATS
from sandbox.projects.antiadblock.qa.tasks.performance_shoot_diff import AntiadblockPerformanceShootsDiff, parse_nginx_access_log
from sandbox.projects.antiadblock.qa.tasks.parameters import BaseProfilingShootParameters
from sandbox.projects.antiadblock.qa.utils.constants import DiffType, FILE_NGINX_ACCESS_LOG_TMPL, CALLGRIND_FILENAME


logger = logging.getLogger("run_task_log")


class AntiadblockProfilingShoot(AntiadblockPerformanceShootsDiff):
    name = 'ANTIADBLOCK_PROFILING_SHOOT'

    class Requirements(sdk2.Task.Requirements):
        privileged = True  # for root
        cores = 6
        disk_space = 80 * 1024  # GB
        ram = 36 * 1024  # GB + 16 GB for cached stub in tmpfs
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(BaseProfilingShootParameters):
        description = 'Antiadblock profiling shoot task'

        with sdk2.parameters.Group('General settings') as general_settings:
            rps = sdk2.parameters.Integer('RPS', default_value=100, required=True)
            shoot_time = sdk2.parameters.Integer('Shoot time in seconds', default_value=60, required=True)
            error_rate_thr = sdk2.parameters.Float('Error rate threshold', default_value=0.003, required=True)
            profile_clock_type = sdk2.parameters.String(
                'Profile clock type',
                choices=[('cpu', 'cpu'), ('wall', 'wall')],
                required=False,
            )

    def check_nginx_logs(self):
        errors = []
        engine_logs_path = getattr(self, 'engine_logs_path', '')
        nginx_access_stat = parse_nginx_access_log(os.path.join(engine_logs_path, FILE_NGINX_ACCESS_LOG_TMPL.format("0")), None, logger)
        # errors
        for error in ('count_server_errors', 'count_client_errors'):
            if nginx_access_stat['all'][error] > float(self.Parameters.error_rate_thr) * nginx_access_stat['all']['count_req']:
                message = 'Too many {} in our nginx access log: {}'.format(error, nginx_access_stat['all'][error])
                errors.append(message)

        if errors:
            raise TaskFailure('\n'.join(errors))

    def save_engine_logs(self, prefix='0'):
        super(AntiadblockProfilingShoot, self).save_engine_logs(prefix)
        logger.info('Try get callgrind from {}'.format(CALLGRIND_FILENAME))
        if os.path.exists(CALLGRIND_FILENAME):
            callgring_file_path = "cryprox_stats_{}.callgrind".format(prefix)
            shutil.copy(CALLGRIND_FILENAME, callgring_file_path)
            callgrind_resource = ANTIADBLOCK_CALLGRIND_STATS(self, 'Callgrind stats for shoot {}'.format(prefix),
                                                             callgring_file_path)
            sdk2.ResourceData(callgrind_resource).ready()
        else:
            logger.error('Callgrind stats is not exists')

    def on_execute(self):
        stub_dir = str(self.ramdrive.path) if self.ramdrive else ''
        self.configs_stub_module = ConfigsStubSandboxAdapter(self.Parameters.configs_stub_parameters, self).create_module()
        with self.configs_stub_module as active_stub:
            self.engine_service = EngineSandboxAdapter(self.Parameters.engine_parameters, self).create_module(stub_dir, DiffType.PROFILING,
                                                                                                              active_stub.get_port(),
                                                                                                              profile_clock_type=self.Parameters.profile_clock_type)
            self.run_pipeline()
            self.engine_service.clean()

        logger.info("Check nginx logs")
        self.check_nginx_logs()
