import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types.misc import DnsType
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
import sandbox.common.types.notification as ctn

from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


class AdblockRuleSonarBin(sdk2.Resource):
    """
    antiadblock/adblock_rule_sonar binary
    """
    releasable = True
    any_arch = True
    auto_backup = True
    group = 'ANTIADBLOCK'
    releasers = ['ANTIADBLOCK']
    release_subscribers = ["ANTIADBLOCK"]


class AAB_RUN_SONAR_BIN(sdk2.Task):
    """ Run antiadblock/adblock_rule_sonar binary from SB resource"""

    class Requirements(sdk2.Requirements):
        dns = DnsType.DNS64  # for external interactions
        client_tags = ctc.Tag.LINUX_XENIAL
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='AAB_RUN_SONAR', weight=1, capacity=1)
            ],
            release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH, ctt.Status.Group.WAIT)
        )

    class Parameters(sdk2.Task.Parameters):
        send_email = sdk2.parameters.Bool('Send email', default=True)
        create_startrack_tickets = sdk2.parameters.Bool('Create StarTrack tickets', default=False)
        with send_email.value[True]:
            email_receiver = sdk2.parameters.String(
                'Email receiver', required=True,
                default='antiadb-sonar@yandex-team.ru,robot-antiadb@yandex-team.ru'
            )
        binary_id = sdk2.parameters.LastReleasedResource(
            'adblock_rule_sonar binary',
            resource_type=AdblockRuleSonarBin,
            state=(ctr.State.READY,),
            required=False
        )

    def on_execute(self):
        binary_id = self.Parameters.binary_id
        if not binary_id:
            binary_id = sdk2.Resource.find(
                resource_type=AdblockRuleSonarBin,
                state=ctr.State.READY
            ).limit(1).first()
        bin_res = sdk2.ResourceData(binary_id)
        yav_secrets = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()
        env = os.environ.copy()
        env['YT_TOKEN'] = yav_secrets["ANTIADBLOCK_YT_TOKEN"]
        env['STARTREK_TOKEN'] = yav_secrets['ANTIADBLOCK_ST_TOKEN']
        env['SONAR_TVM_SECRET'] = yav_secrets['ANTIADBLOCK_SANDBOX_MONITORING_TVM_SECRET']
        create_tickets_param = ['--create_tickets'] if self.Parameters.create_startrack_tickets else []
        cmd = [str(bin_res.path), '--send_rules'] + create_tickets_param
        with sdk2.helpers.ProcessLog(self, logger="run_sonar_log") as log:
            sp.check_call(cmd, stdout=log.stdout, stderr=log.stderr, env=env)

        if self.Parameters.send_email:
            with open('new_rules.txt', 'r') as output:
                is_urgent = output.readline().startswith('urgent')
                message = output.read().strip()
                if message:
                    self.server.notification(
                        subject='New Adblock rules',
                        body=message,
                        recipients=self.Parameters.email_receiver.split(','),
                        transport=ctn.Transport.EMAIL,
                        urgent=is_urgent,
                    )
