import os
import json
import logging
from copy import deepcopy
from urlparse import urljoin

import requests


ALERT_CUSTOM_PARAMS = dict(
    fraud={'annotations': {'service': '{{labels.service_id}}_{{labels.sensor}}'}},
    errors_4XX_per_dc={'annotations': {'service': '{{labels.host}}'}},
    errors_5XX_per_dc={'annotations': {'service': '{{labels.host}}'}},
)

DEFAULT_ANNOTATIONS = {'service': '{{labels.service_id}}'}
DEFAULT_CHANNELS = [
    {
      "id": "antiadb_zfp_juggler",
      "config": {},
    }
]


def get_secs_from_period(period):
    """
    >> > [get_secs_from_period(n) for n in [1s, 10s, 1m, 15m, 1h, 3h, 1d, 5d, 1w, 4w]]
    [1, 10, 60, 900, 3600, 10800, 86400, 432000, 604800, 2419200]
    """
    multiplier_map = {
        's': 1,
        'm': 60,
        'h': 3600,
        'd': 86400,
        'w': 604800,
    }
    value = int(period[:-1])
    _type = period[-1:]
    return value * multiplier_map[_type]


def get_group_by_labels(multi_sel):
    try:
        return sorted([expr.split('=')[0] for expr in multi_sel.split(',') if expr])
    except Exception as e:
        logging.error(e)
        return []


def get_empty_alert_params(alert_id):
    return {
        'id': alert_id,
        'projectId': 'Antiadblock',
        'name': alert_id,
        'channels': [],
        'type': {'expression': {}},
        'annotations': {},
        'windowSecs': get_secs_from_period('7d'),
        'delaySeconds': 0,
        'description': alert_id,
    }


def update_alert_params(alert_id, alert_params, program, task):
    alert_params['type']['expression']['program'] = program
    alert_params['type']['expression']['checkExpression'] = 'status'

    alert_params['groupByLabels'] = get_group_by_labels(task['description'].get('multi_sel', ''))

    annotations = deepcopy(DEFAULT_ANNOTATIONS)
    annotations['host'] = alert_id
    alert_custom_params = ALERT_CUSTOM_PARAMS.get(alert_id, {})
    annotations.update(alert_custom_params.get('annotations', {}))
    alert_params['annotations'] = annotations
    alert_params['windowSecs'] = get_secs_from_period(task.get('period', '7d'))
    alert_params['periodMillis'] = 1000 * alert_params['windowSecs']
    alert_params['delaySeconds'] = task.get('delaySeconds') or alert_custom_params.get('delay') or 0
    alert_params['channels'] = DEFAULT_CHANNELS
    return alert_params


def update_alert(alert_id, abs_path, solomon_token, use_prestable=False):
    logging.info("Update alert: {}".format(alert_id))
    SOLOMON_BASE_URL = 'solomon-prestable.yandex-team.ru' if use_prestable else 'solomon.yandex-team.ru'
    SOLOMON_API_BASE_URL = 'https://{}/api/v2/projects/Antiadblock/'.format(SOLOMON_BASE_URL)
    SOLOMON_HEADERS = {'Authorization': 'OAuth {}'.format(solomon_token),
                       'Content-Type': 'application/json', 'Accept': 'application/json'}
    SOLOMON_ALERTS_API_URL = urljoin(SOLOMON_API_BASE_URL, 'alerts/')

    path_to_program = os.path.join(abs_path, alert_id, 'program.saql')
    with open(path_to_program) as program_file:
        program = program_file.read()

    path_to_task = os.path.join(abs_path, alert_id, 'task.json')
    with open(path_to_task) as task_file:
        task = json.loads(task_file.read())

    session = requests.Session()
    session.headers.update(SOLOMON_HEADERS)

    action = ""
    response = session.get(urljoin(SOLOMON_ALERTS_API_URL, alert_id))
    if response.status_code == 200:
        alert_params = response.json()
        response = session.put(urljoin(SOLOMON_ALERTS_API_URL, alert_id), json=update_alert_params(alert_id, alert_params, program, task))
        action = "Update"
    elif response.status_code == 404:
        response = session.post(SOLOMON_ALERTS_API_URL, json=update_alert_params(alert_id, get_empty_alert_params(alert_id), program, task))
        action = "Create"

    response.raise_for_status()
    logging.info("{} alert success: {}".format(action, alert_id))
