import logging
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.errors import TaskFailure

from sandbox.projects.antiadblock.zfp import AabZfpPackage
from sandbox.projects.antiadblock.utils import ROBOT_ANTIADB_TOKENS_YAV_ID


TEMPLATE_WORKFLOW_ID = '10af1c4c-da68-4391-91f7-4e941e0efe1e'
TARGET_WORKFLOW_ID = '5460eda9-0cde-48da-a29e-fdb048969c8b'

ARCADIA_PATH_TEMPLATE = 'antiadblock/tasks/monitoring_update/programs/{}/{}'

INSTANCE_TEMPLATE_URL = 'https://nirvana.yandex-team.ru/flow/{}/{}/graph'


class AntiadblockZfpRunCalculation(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        zfp_package_resource = sdk2.parameters.Resource(
            'Package with alert programs and aggregate configs for ZFP test running',
            resource_type=AabZfpPackage)

        alert_ids = sdk2.parameters.List('alert ids to run calculation', required=True)
        juggler_hosts = sdk2.parameters.List('juggler hosts to run calculation for', required=True)
        days = sdk2.parameters.Integer('count days for calculation', default_value=45, required=True)
        trace_instances = sdk2.parameters.Bool("Trace instances", default=True)
        with trace_instances.value[True]:
            retry_period = sdk2.parameters.Integer(
                "Time period to check request status (in seconds)",
                default=60 * 5
            )

        debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
        with debug_mode.value[True]:
            period_from = sdk2.parameters.String("Start datetime in format %Y-%m-%d %H:%M:%S", default_value="")
            period_to = sdk2.parameters.String("Finish datetime in format %Y-%m-%d %H:%M:%S", default_value="")

        with sdk2.parameters.Output:
            instances = sdk2.parameters.JSON('Nirvana instance urls', default={})
            is_finished = sdk2.parameters.Bool('Nirvana instances is finished ', default=False)

    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment('nirvana_api')]

    class Context(sdk2.Task.Context):
        is_finished = {}

    def on_execute(self):

        instances = {}
        with self.memoize_stage.send_request:
            for host in self.Parameters.juggler_hosts:
                instance_id = self._create_pipeline_instance(host)
                instances[host] = INSTANCE_TEMPLATE_URL.format(TARGET_WORKFLOW_ID, instance_id)
                self.Context.is_finished[instance_id] = False

            self.Parameters.instances = instances

        if self.Parameters.trace_instances:
            workflow_execution_states = self._get_instances_status()
            for state in workflow_execution_states:
                logging.info(state)

            if any(state['status'] != 'completed' for state in workflow_execution_states):
                self._sleep()

            if any(state['result'] != 'success' for state in workflow_execution_states):
                raise TaskFailure("Eval graphs didn't complete successfully")

            self.Parameters.is_finished = True

    def _sleep(self):
        raise sdk2.WaitTime(self.Parameters.retry_period)

    def _get_instances_status(self):
        import nirvana_api
        NIRVANA_OAUTH_TOKEN = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()["ANTIADBLOCK_NIRVANA_TOKEN"]
        API = nirvana_api.NirvanaApi(oauth_token=NIRVANA_OAUTH_TOKEN, ssl_verify=False)
        return API.get_workflow_execution_states(self.Context.is_finished.keys())

    def _create_pipeline_instance(self, juggler_host):
        import nirvana_api
        NIRVANA_OAUTH_TOKEN = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()["ANTIADBLOCK_NIRVANA_TOKEN"]
        API = nirvana_api.NirvanaApi(oauth_token=NIRVANA_OAUTH_TOKEN, ssl_verify=False)
        template_instance_id = API.find_workflows(pattern=TEMPLATE_WORKFLOW_ID)[0].instanceId

        target_instance_id = API.clone_workflow_instance(
            workflow_id=TEMPLATE_WORKFLOW_ID,
            workflow_instance_id=template_instance_id,
            target_workflow_id=TARGET_WORKFLOW_ID,
        )
        if not self.Parameters.debug_mode:
            period_to = datetime.now()
            period_from = period_to - timedelta(days=int(self.Parameters.days))
        else:
            period_to = datetime.strptime(self.Parameters.period_to, "%Y-%m-%d %H:%M:%S")
            period_from = datetime.strptime(self.Parameters.period_from, "%Y-%m-%d %H:%M:%S")
        param_values = [nirvana_api.ParameterValue('alert_ids', self.Parameters.alert_ids),
                        nirvana_api.ParameterValue('juggler_host', juggler_host),
                        nirvana_api.ParameterValue('period_to', period_to.strftime('%s')),
                        nirvana_api.ParameterValue('period_from', period_from.strftime('%s')),
                        nirvana_api.ParameterValue('zfp_package_resource_id', str(self.Parameters.zfp_package_resource.id))]

        API.set_global_parameters(
            workflow_id=TARGET_WORKFLOW_ID,
            workflow_instance_id=target_instance_id,
            param_values=param_values,
        )

        instance_id = API.start_workflow(
            workflow_id=TARGET_WORKFLOW_ID,
            workflow_instance_id=target_instance_id,
        )
        return instance_id
