# coding=utf-8

import re
import logging
from urllib import quote
from datetime import datetime, timedelta
from itertools import product
from collections import defaultdict

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task

JUGGLER_HOSTS = [
    'antiadb_4xx_errors', 'antiadb_403_errors', 'antiadb_5xx_errors', 'antiadb_599_errors', 'antiadb_rps_errors', 'antiadb_bamboozled',
    'antiadb_check_uid_cookies', 'antiadb_headers_x_real_ip', 'antiadb_headers_x_forwarded_for', 'antiadb_headers_present',
    'antiadb_fraud_bad_shows', 'antiadb_fraud_bad_clicks', 'antiadb_fraud_bad_money',
    'antiadb_fraud_fraud_shows', 'antiadb_fraud_fraud_clicks', 'antiadb_fraud_fraud_money',
]

SERVICE_IDS = [
    'autoredirect.turbo', 'autoru', 'docviewer.yandex.ru', 'drive2', 'echomsk',
    'games.yandex.ru', 'gorod_rabot', 'kinopoisk.ru', 'liveinternet', 'livejournal',
    'nova.rambler.ru', 'otzovik', 'razlozhi', 'sdamgia', 'smi24', 'turbo.yandex.ru',
    'yandex_afisha', 'yandex_images', 'yandex_mail', 'yandex_morda',
    'yandex_news', 'yandex_player', 'yandex_pogoda', 'yandex_realty', 'yandex_sport',
    'yandex_tv', 'yandex_video', 'zen.yandex.ru',
]

DATE_FMT = '%Y-%m-%d %H:%M:00'

GRAPH_URL_RE = re.compile(r'.*(https://solomon.*?)\s')
STARTREK_URL_TEMPLATE = 'https://st.yandex-team.ru/createTicket?queue=ANTIADBALERTS&priority=2&summary={}%3A{}&type=2&tags[0]=false_positive&incidentStart={}'
ROBOT_ANTIADB_TOKENS_YAV_ID = "sec-01crfs4yjd29hardw658g1qk84"


class AabZfpCheckAlerts(binary_task.LastBinaryTaskRelease, sdk2.Task):
    name = "AAB_ZFP_CHECK_ALERTS"

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup("Checks juggler_host", required=True) as juggler_hosts:
            juggler_hosts.choices = [(_, _) for _ in JUGGLER_HOSTS]

        with sdk2.parameters.CheckGroup("Checks service_id", required=True) as service_ids:
            service_ids.choices = [(_, _) for _ in SERVICE_IDS]

        datetime_from = sdk2.parameters.String(
            'Datetime from', required=True,
            default_value=(datetime.now() - timedelta(days=1)).strftime(DATE_FMT)
        )
        datetime_to = sdk2.parameters.String(
            'Datetime to', required=True,
            default_value=datetime.now().strftime(DATE_FMT)
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        pass

    @sdk2.report(title="Alerts report")
    def report(self):
        return self.Context.report or "No report discovered in context"

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": AabZfpCheckAlerts.name, "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        checks = list(product(self.Parameters.juggler_hosts, self.Parameters.service_ids))
        juggler_checks = self.get_juggler_checks(checks)

        report = "<h2><b>Интервал проверки: {} - {}</b></h2>\n".format(self.Parameters.datetime_from, self.Parameters.datetime_to)
        report += self.render_report(juggler_checks)
        self.Context.report = report
        self.Context.save()

    def get_juggler_checks(self, checks):
        from antiadblock.tasks.tools.juggler import get_check_history
        result = defaultdict(list)
        since = datetime.strptime(self.Parameters.datetime_from, DATE_FMT)
        until = datetime.strptime(self.Parameters.datetime_to, DATE_FMT)
        issues = self.get_all_issue(checks, since, until)
        logging.debug(issues)
        for host, service in checks:
            juggler_checks = get_check_history(dict(host=host, service=service), since.strftime('%s'), until.strftime('%s'), raw=True)
            logging.debug(juggler_checks)
            exists_ts = set()
            for check in sorted(juggler_checks, key=lambda el: el['status_time']):
                description = check.get('description', '')
                ts = check['status_time']
                ts = ts - ts % 60
                if ts in issues.get((host, service), set()):
                    continue
                if ts not in exists_ts:
                    exists_ts.add(ts)
                    graph_url = self.get_graph_url(description, since, until)
                    create_issue_url = self.get_create_issue_url(host, service, ts)
                    result["{}:{}".format(host, service)].append(dict(ts=ts, graph_url=graph_url, create_issue_url=create_issue_url))

        return result

    @staticmethod
    def get_graph_url(description, since, until):
        from antiadblock.tasks.tools.common_configs import SOLOMON_TS_I_FMT, MSK_UTC_OFFSET
        match = GRAPH_URL_RE.search(description)
        if match:
            b = quote((since - MSK_UTC_OFFSET - timedelta(hours=4)).strftime(SOLOMON_TS_I_FMT))
            e = quote((until - MSK_UTC_OFFSET).strftime(SOLOMON_TS_I_FMT))
            return "{}&b={}&e={}".format(match.group(1), b, e)
        return ''

    @staticmethod
    def get_create_issue_url(host, service, ts):
        from antiadblock.tasks.tools.common_configs import STARTREK_DATETIME_FMT, MSK_UTC_OFFSET
        ticket_link = STARTREK_URL_TEMPLATE.format(host, service, (datetime.fromtimestamp(ts) - MSK_UTC_OFFSET).strftime(STARTREK_DATETIME_FMT))
        return ticket_link

    @staticmethod
    def render_report(checks):
        style = """<style>
            table, th, td {
                border: 1px solid grey;
                border-collapse: collapse;
                padding: 5px;
            }
        </style>\n"""
        content = ""
        for key, val in checks.items():
            content += '<b>{}</b><br />\n'.format(key)
            content += '<table>\n'
            content += '''<tr><th width="150px">Время</th><th width="200px">Ссылка на график</th><th width="200px">Тикет</th></tr>\n'''
            for check in val:
                content += '''<tr><td>{}</td><td><a href="{}" target="_blank">График</a></td>
                <td><a href="{}" target="_blank">Создать тикет</a></td></tr>\n'''.format(
                    datetime.fromtimestamp(check['ts']).strftime(DATE_FMT), check['graph_url'], check['create_issue_url']
                )
            content += '</table>\n'

        return style + content

    def get_all_issue(self, checks, since, until):
        from startrek_client import Startrek
        token = sdk2.yav.Secret(ROBOT_ANTIADB_TOKENS_YAV_ID).data()['ANTIADBLOCK_ST_TOKEN']
        st_client = Startrek(useragent='python', token=token)
        result = {}
        for host, service in checks:
            result[(host, service)] = self.get_st_history(st_client, host, service, since, until)

        return result

    @staticmethod
    def get_st_history(client, host, service, since, until):
        from antiadblock.tasks.tools.common_configs import STARTREK_DATETIME_FMT, MSK_UTC_OFFSET
        since = (since - MSK_UTC_OFFSET).strftime(STARTREK_DATETIME_FMT).split('T')[0]
        until = (until - MSK_UTC_OFFSET).strftime(STARTREK_DATETIME_FMT).split('T')[0]
        query = '''Queue: ANTIADBALERTS Summary: "{}:{}" "Incident start": >= "{}" "Incident start": <= "{}"'''.format(
            host, service, since, until,
        )
        logging.debug(query)
        issues = client.issues.find(query=query)
        logging.debug(issues)
        result = set()
        for issue in issues:
            assert len(issue.tags) == 1
            assert issue.tags[0] == "false_positive"
            incident_start = issue.incidentStart
            dt = datetime.strptime(incident_start.split('.')[0], STARTREK_DATETIME_FMT) + MSK_UTC_OFFSET
            dt = int(dt.strftime('%s'))
            result.add(dt)
        return result
