# -*- coding: utf-8 -*-

import json
import os

from collections import defaultdict, namedtuple

from sandbox import sdk2
from sandbox.sandboxsdk.parameters import SandboxInfoParameter, SandboxBoolParameter

from sandbox.common.errors import ReleaseError
from sandbox.common.types import client as ctc

from sandbox.projects.common.build import YaPackage, parameters as build_params
from sandbox.projects.common.z2.z2_client import Z2Client

from sandbox.projects.antifraud.tasks import (
    YA_VAULT_Z2_ANTIFRAUD_SECRET,
    set_tests_info_and_get_brief,
)
from sandbox.projects.antifraud.resources import (
    AntifraudAnalyticsDebianPackage,
    AntifraudCleaningDebianPackage,
    AntifraudDataDebianPackage,
    AntifraudProgramsDebianPackage,
    AntifraudScriptsDebianPackage,
)


DEBIAN_PACKAGES_PATH = "quality/antifraud/deploy/debian"
YA_PACKAGE_FILE_NAME = "pkg.json"
ANTIFRAUD_ROBOT_LOGIN = "robot-antifraud"

PackageInfo = namedtuple("PackageInfo", ["resource_type", "context_param", "z2_config_ids"])
PACKAGE_INFOS = {
    "yandex-search-antifraud-analytics": PackageInfo(
        resource_type=AntifraudAnalyticsDebianPackage.name,
        context_param="build_yandex_search_antifraud_analytics",
        z2_config_ids=["ANTIFRAUD_DEV"],
    ),
    "yandex-search-antifraud-cleaning": PackageInfo(
        resource_type=AntifraudCleaningDebianPackage.name,
        context_param="build_yandex_search_antifraud_cleaning",
        z2_config_ids=["MR_VELES02"],
    ),
    "yandex-search-antifraud-data": PackageInfo(
        resource_type=AntifraudDataDebianPackage.name,
        context_param="build_yandex_search_antifraud_data",
        z2_config_ids=["AFRAUDITA", "MR_VELES02"],
    ),
    "yandex-search-antifraud-programs": PackageInfo(
        resource_type=AntifraudProgramsDebianPackage.name,
        context_param="build_yandex_search_antifraud_programs",
        z2_config_ids=["AFRAUDITA"],
    ),
    "yandex-search-antifraud-scripts": PackageInfo(
        resource_type=AntifraudScriptsDebianPackage.name,
        context_param="build_yandex_search_antifraud_scripts",
        z2_config_ids=["AFRAUDITA"],
    ),
}


class DebianPackagesBlock(SandboxInfoParameter):
    description = "Debian packages"


class DebianPackagesParameters(sdk2.Task.Parameters):
    for package, package_info in PACKAGE_INFOS.items():
        sdk2.helpers.set_parameter(
            package_info.context_param,
            sdk2.parameters.Bool("Build and release {}.deb".format(package)),
        )
debian_packages_parameters = build_params.get_sdk1_parameters(DebianPackagesParameters)


class RunLongTestsParameter(SandboxBoolParameter):
    name = "run_long_tests"
    description = "Run long tests"
    required = False
    default_value = True


class IgnoreFailTestsParameter(SandboxBoolParameter):
    name = "ignore_fail_tests"
    description = "Ignore fail tests"
    required = False
    default_value = True


class RunTestsParameter(SandboxBoolParameter):
    name = "run_tests"
    description = "Run tests after build"
    required = True
    default_value = True
    sub_fields = {
        "true": [
            RunLongTestsParameter.name,
            IgnoreFailTestsParameter.name
        ]
    }


class AntifraudBuildDebianPackages(YaPackage.YaPackage, object):
    """Task for creating antifraud debian packages.
    """
    type = "ANTIFRAUD_BUILD_DEBIAN_PACKAGES"
    execution_space = 120 * 1024
    required_ram = 100 * 1024
    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [
        YaPackage.parameters.ArcadiaUrl,

        DebianPackagesBlock,
    ] + debian_packages_parameters + [

        # Tests
        YaPackage.TestsBlock,
        RunTestsParameter,
        RunLongTestsParameter,
        IgnoreFailTestsParameter,
    ]

    def _z2_post(self, z2_client, action, payload=None):
        z2_payload = z2_client.init_payload()
        z2_payload.update(payload or {})
        response = z2_client.request(action, z2_payload, "POST")
        response_data = json.loads(response.text)
        success = response_data.get("success", False)
        if not success:
            err = response_data.get("errorMsg", "unknown")
            raise ReleaseError("Z2 API error: {}".format(err))

    def _z2_update(self, z2_client, z2_config_id, z2_packages):
        self.set_info("Edit packages {} in Z2 config id '{}'".format(z2_packages, z2_config_id))
        self._z2_post(z2_client, "editItems", {"items": json.dumps(z2_packages)})
        self.set_info("Updating Z2 config id '{}'".format(z2_config_id))
        self._z2_post(z2_client, "update")

    def on_enqueue(self):
        self.ctx[YaPackage.PackageTypeParameter.name] = YaPackage.DEBIAN
        self.ctx[YaPackage.CompressPackageArchiveParameter.name] = True
        self.ctx[YaPackage.StripBinariesParameter.name] = False
        self.ctx[YaPackage.PublishPackageParameter.name] = True
        self.ctx[YaPackage.KeyUserParameter.name] = ANTIFRAUD_ROBOT_LOGIN
        self.ctx[YaPackage.PublishToParameter.name] = "search"

        packages = []
        resource_types = []
        for package, package_info in PACKAGE_INFOS.items():
            if not self.ctx.get(package_info.context_param):
                continue
            package_path = os.path.join(DEBIAN_PACKAGES_PATH, package, YA_PACKAGE_FILE_NAME)
            packages.append(package_path)
            resource_types.append(package_info.resource_type)

        self.ctx[YaPackage.PackagesParameter.name] = ";".join(packages)
        self.ctx[YaPackage.ResourceTypeParameter.name] = ";".join(resource_types)

        self.ctx[YaPackage.parameters.BuildType.name] = YaPackage.RELEASE
        self.ctx[YaPackage.parameters.UseArcadiaApiFuse.name] = True
        self.ctx[YaPackage.parameters.UseArcInsteadOfArcadiaApi.name] = True
        self.ctx[YaPackage.parameters.AllowArcadiaApiFallback.name] = True

        YaPackage.YaPackage.on_enqueue(self)

    def on_execute(self):
        YaPackage.YaPackage.on_execute(self)
        tests_brief = set_tests_info_and_get_brief(self, "junit*.xml")
        self.descr = "<br><br>".join([tests_brief, self.descr])

    def on_release(self, additional_parameters):
        self.mark_released_resources(additional_parameters["release_status"])
        secret_data = sdk2.yav.Secret(YA_VAULT_Z2_ANTIFRAUD_SECRET).data()

        packages_by_z2 = defaultdict(list)
        for resource in self.list_resources():
            if resource.is_ready() and resource.type.releasable:
                resource.attrs["ttl"] = 365  # 1 year
                package = resource.attrs["resource_name"]
                package_info = PACKAGE_INFOS[package]
                for z2_config_id in package_info.z2_config_ids:
                    packages_by_z2[z2_config_id].append({
                        "name": package,
                        "version": resource.attrs["resource_version"],
                    })

        for z2_config_id, z2_packages in packages_by_z2.items():
            if z2_config_id not in secret_data:
                raise ReleaseError("Z2 config id '{}' is not found in yav secret: {}".format(
                    z2_config_id,
                    YA_VAULT_Z2_ANTIFRAUD_SECRET,
                ))
            z2_client = Z2Client(secret_data[z2_config_id], z2_config_id, dry_run=False)
            self._z2_update(z2_client, z2_config_id, z2_packages)
