# -*- coding: utf-8 -*-

import glob
import os
import re

from lxml import etree

from sandbox.common.types import client as ctc
from sandbox.projects.common.build import YaPackage
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter


YA_VAULT_ROBOT_ANTIFRAUD_SECRET = "sec-01dcrpzkqqjwvxqfk1rdc61s0d"
YA_VAULT_ROBOT_ANTIFRAUD_NIRVANA_TOKEN_KEY = "nirvana"
YA_VAULT_ROBOT_ANTIFRAUD_SSH_KEY = "ssh-key"

YA_VAULT_Z2_ANTIFRAUD_SECRET = "sec-01eackrmmbsftr3rkd87s3dz03"
ROBOT_ANTIFRAUD_SSH_LOGIN = "robot-antifraud"


class PackagesParameter(SandboxStringParameter):
    name = "packages"
    description = "Package path"
    required = True


class ResourceTypeParameter(SandboxStringParameter):
    name = "resource_type"
    description = "Resource type"
    required = True


class RunLongTestsParameter(SandboxBoolParameter):
    name = "run_long_tests"
    description = "Run long tests"
    required = False
    default_value = True


class IgnoreFailTestsParameter(SandboxBoolParameter):
    name = "ignore_fail_tests"
    description = "Ignore fail tests"
    required = False
    default_value = True


class RunTestsParameter(SandboxBoolParameter):
    name = "run_tests"
    description = "Run tests after build"
    required = True
    default_value = True
    sub_fields = {
        "true": [
            RunLongTestsParameter.name,
            IgnoreFailTestsParameter.name
        ]
    }


class AntifraudBuildBaseTask(YaPackage.YaPackage, object):
    """
        Task for creating tarball package
    """
    type = "ANTIFRAUD_BUILD_BASE"
    execution_space = 70 * 1024
    required_ram = 10 * 1024
    client_tags = ctc.Tag.Group.LINUX & ctc.Tag.SSD

    input_parameters = [
        YaPackage.parameters.ArcadiaUrl,
        YaPackage.parameters.BuildType,
        YaPackage.parameters.ArcadiaPatch,
        YaPackage.parameters.UseArcadiaApiFuse,
        # Package
        YaPackage.PackagesBlock,
        PackagesParameter,
        # Resources
        YaPackage.ResourcesBlock,
        ResourceTypeParameter,
        # Tests
        YaPackage.TestsBlock,
        RunTestsParameter,
        RunLongTestsParameter,
        IgnoreFailTestsParameter,
    ]

    def on_enqueue(self):
        self.ctx[YaPackage.PackageTypeParameter.name] = YaPackage.TARBALL
        self.ctx[YaPackage.parameters.BuildType.name] = YaPackage.RELEASE
        self.ctx[YaPackage.CompressPackageArchiveParameter.name] = True
        self.ctx[YaPackage.StripBinariesParameter.name] = False
        YaPackage.YaPackage.on_enqueue(self)

    def on_execute(self):
        YaPackage.YaPackage.on_execute(self)
        tests_brief = set_tests_info_and_get_brief(self, "junit*.xml")
        self.descr = "<br><br>".join([self.descr, tests_brief])

    def on_release(self, additional_parameters):
        super(AntifraudBuildBaseTask, self).on_release(additional_parameters)
        for resource in self.list_resources():
            if resource.type.releasable:
                resource.attrs["ttl"] = 777  # > 2 years for recalcs


def process_junit_report(path):
    report = etree.parse(path)
    cases = int(report.xpath("count(//testsuites/testsuite/testcase)"))
    failures = int(report.xpath("count(//testsuites/testsuite/testcase/failure)"))
    if failures == 0:
        return cases, 0, None

    tmpl = R"""
        <xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
        <xsl:template match="/">
          <div>
            <h4>Tests info</h4>
            <div style="color: red; font-size: 1.2rem;">Failed tests: <xsl:value-of select="count(testsuites/testsuite/testcase/failure)"/>.</div>
            <div>
              <xsl:for-each select="testsuites/testsuite/testcase">
                <xsl:if test="count(failure) > 0">
                  <hr />
                  Testcase "<xsl:value-of select="@name"/>" failures:
                  <xsl:for-each select="failure">
                    <div>
                      <xsl:value-of select="."/>
                    </div>
                  </xsl:for-each>
                </xsl:if>
              </xsl:for-each>
            </div>
          </div>
        </xsl:template>
        </xsl:stylesheet>
    """
    tmpl = re.sub("\n *", "", tmpl)
    xslt = etree.fromstring(tmpl)
    xslt_transform = etree.XSLT(xslt)
    result = xslt_transform(report)
    return cases, failures, etree.tostring(result)


def set_tests_info_and_get_brief(task_obj, junit_report_glob):
    total_cases = total_failures = 0
    for junit_report_file in glob.glob(junit_report_glob):
        if not os.path.exists(junit_report_file):
            continue
        cases, failures, info = process_junit_report(junit_report_file)
        if info is not None:
            task_obj.set_info(info, do_escape=False)
        total_failures += failures
        total_cases += cases
    message_lines = []
    if total_cases:
        message_lines.append(
            'Junit reports contain {} test cases.'
            .format(total_cases)
        )
        if total_failures:
            message_lines.append(
                'There are <span style="color: red;">{} failed tests</span>.'
                .format(total_failures)
            )
            message_lines.append('See detailed information below.')
        else:
            message_lines.extend([
                'There are <span style="color: #00CC00;">no failed tests</span>.',
                'Good job!',
            ])
    else:
        message_lines.append('<span style="color: blue;">No test cases</span> were found.')
    return "<br>".join(message_lines)
