# -*- coding: utf-8 -*-

import base64
import os
import shutil
import stat

from sandbox.sdk2 import yav
from sandbox.projects.antifraud.resources import AntifraudNirvactorTool, AntifraudReactorDeploySpec
from sandbox.sandboxsdk.parameters import ResourceSelector, SandboxStringParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import check_process_return_code, run_process

from sandbox.projects.antifraud.tasks import (
    YA_VAULT_ROBOT_ANTIFRAUD_SECRET,
    YA_VAULT_ROBOT_ANTIFRAUD_NIRVANA_TOKEN_KEY,
    YA_VAULT_ROBOT_ANTIFRAUD_SSH_KEY,
    ROBOT_ANTIFRAUD_SSH_LOGIN,
)


REACTOR_RESOURCE_SPEC_CONTEXT_KEY = '_reactor_resource_spec'
REACTOR_SECURE_SPEC_CONTEXT_KEY = '_reactor_secure_spec'


class NirvactorToolParameter(ResourceSelector):
    name = 'nirvactor_tool_resource'
    description = 'Nirvactor tool resource'
    resource_type = AntifraudNirvactorTool
    default_value = None


class ReactorResourceSpecParameter(SandboxStringParameter):
    name = 'reactor_resource_spec_path'
    description = 'Reactor spec to deploy resource'
    default_value = None


class ReactorSecureSpecParameter(SandboxStringParameter):
    name = 'reactor_secure_spec_path'
    description = 'Reactor spec to deploy data from secure svn repo'
    default_value = None


class AntifraudReactorDeploy(object):

    input_parameters = [
        NirvactorToolParameter,
        ReactorResourceSpecParameter,
        ReactorSecureSpecParameter,
    ]

    def _get_latest_resouce(self, **kwargs):
        resources = channel.sandbox.list_resources(order_by='-id', limit=1, status='READY', **kwargs)
        if not resources:
            raise RuntimeError('Cannot find ready resources: {}'.format(kwargs))
        return resources[0]

    def _make_spec_resource(self, arcadia_spec_path, context_key):
        self.set_info(
            'Creating resource with deploy spec for "{}" context key: {}'
            .format(context_key, arcadia_spec_path)
        )
        spec_resource_path = self.path(context_key + '.yaml')
        with self.get_arcadia(use_fuse=True, checkout=False) as arcadia_src_dir:
            shutil.copyfile(os.path.join(arcadia_src_dir, arcadia_spec_path), spec_resource_path)

        spec_resource = self.create_resource('Reactor spec', spec_resource_path, AntifraudReactorDeploySpec)
        self.mark_resource_ready(spec_resource)
        self.ctx[context_key] = spec_resource.id

    def _run_nirvactor_tool(self, command):
        secret = yav.Secret(YA_VAULT_ROBOT_ANTIFRAUD_SECRET)
        nirvana_token = secret.data()[YA_VAULT_ROBOT_ANTIFRAUD_NIRVANA_TOKEN_KEY]

        self.set_info('Running deploy process: {}'.format(command))
        process = run_process(
            command,
            shell=True,
            check=False,
            log_prefix='deploy',
            environment={'NIRVANA_TOKEN': nirvana_token},
        )
        with open(process.stdout_path) as fp:
            deploy_log = fp.read()
        self.set_info('Deploy log:\n{}'.format(deploy_log))
        check_process_return_code(process)

    def _release_resource(self, nirvactor_tool, spec_resource_id, additional_parameters):
        self.mark_released_resources(additional_parameters['release_status'])

        release_info = self.get_release_info()
        release_resources = release_info['release_resources']
        if len(release_resources) != 1:
            raise RuntimeError('Cannot release more than one resource: {}'.format(release_resources))

        release_resource = self.sync_resource(release_resources[0]['id'])
        reactor_resource_spec = self.sync_resource(spec_resource_id)

        self._run_nirvactor_tool([
            nirvactor_tool, 'deploy-package',
            '--package', release_resource,
            '--spec', reactor_resource_spec,
        ])

    def _release_secure(self, nirvactor_tool, spec_resource_id):
        secret = yav.Secret(YA_VAULT_ROBOT_ANTIFRAUD_SECRET)
        ssh_key_b64 = secret.data()[YA_VAULT_ROBOT_ANTIFRAUD_SSH_KEY]
        ssh_key = base64.b64decode(ssh_key_b64)
        ssh_key_path = self.path('ssh_key')
        with open(ssh_key_path, 'w') as fp:
            fp.write(ssh_key)
        os.chmod(ssh_key_path, stat.S_IRUSR)

        reactor_secure_spec = self.sync_resource(spec_resource_id)
        self._run_nirvactor_tool([
            nirvactor_tool, 'deploy-secure',
            '--ssh-key', ssh_key_path,
            '--ssh-user', ROBOT_ANTIFRAUD_SSH_LOGIN,
            '--spec', reactor_secure_spec,
        ])

    def on_execute(self):
        if self.ctx.get(NirvactorToolParameter.name) is None:
            nirvactor_tool_resource = self._get_latest_resouce(resource_type=AntifraudNirvactorTool)
            self.ctx[NirvactorToolParameter.name] = nirvactor_tool_resource.id

        context_key_by_parameter = {
            REACTOR_RESOURCE_SPEC_CONTEXT_KEY: ReactorResourceSpecParameter.name,
            REACTOR_SECURE_SPEC_CONTEXT_KEY: ReactorSecureSpecParameter.name,
        }

        for context_key, parameter_name in context_key_by_parameter.iteritems():
            arcadia_spec_path = self.ctx.get(parameter_name)
            if arcadia_spec_path is not None:
                self._make_spec_resource(arcadia_spec_path, context_key)
        super(AntifraudReactorDeploy, self).on_execute()

    def on_release(self, additional_parameters):
        nirvactor_tool = self.sync_resource(self.ctx.get(NirvactorToolParameter.name))

        resource_spec = self.ctx.get(REACTOR_RESOURCE_SPEC_CONTEXT_KEY)
        if resource_spec is not None:
            self._release_resource(nirvactor_tool, resource_spec, additional_parameters)

        secure_spec = self.ctx.get(REACTOR_SECURE_SPEC_CONTEXT_KEY)
        if secure_spec is not None:
            self._release_secure(nirvactor_tool, secure_spec)
