# -*- coding: utf-8 -*-

import logging
import os
import re
import socket
import time

from sandbox import sdk2
from sandbox.sandboxsdk import copy
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc
from sandbox.projects.resource_types import ANTIROBOT_BUNDLE


class AntirobotLoadTestingResource(sdk2.Resource):
    """
    Load testing artifacts
    """
    any_arch = True
    executable = False
    releasable = False


CAPTCHA_MOCK_PORT = 10808
WIZARD_MOCK_PORT = 8891
ANTIROBOT_CACHER_PORT = 8899
ANTIROBOT_PROCESSOR_PORT = 8898
ANTIROBOT_ADMIN_PORT = 8897
DOLBILKA_RPS = 1000
DOLBILKA_REQUESTS_LIMIT = 60000

RUN_ANTIROBOT_CMD = "./antirobot_daemon -c antirobot.lua --no-tvm"
RUN_CAPTCHA_MOCK_CMD = "./api_captcha_mock -p %d" % CAPTCHA_MOCK_PORT
RUN_WIZARD_MOCK_CMD = "./wizard_mock %d" % WIZARD_MOCK_PORT
RUN_DOLBILKA_EXECUTE_CMD = "./d-executor -Q {request_limit} -R {rps} -cp plan.bin -o dump.bin -P {shoot_port}"
RUN_DOLBILKA_DUMPER_CMD = "./d-dumper -a -f dump.bin"

DOLBILKA_BINARIES_RBTORRENT = "rbtorrent:669f3d5c459d8383d6ea835661b3745a394e3153"
DOLBILKA_PLAN_RBTORRENT = "rbtorrent:9a88de6c43fa93efee2ba8f2681d3d02ce26414c"
MOCKS_RBTORRENT = "rbtorrent:f243dfcb632578cbd841e0fd9b19d32fa6b82ae8"
ANTIROBOT_FORMULAS_RBTORRENT = "rbtorrent:4f26142909e5f906c117e15a7b2239333ce166ea"
ANTIROBOT_DATA_RBTORRENT = "rbtorrent:ed558fdbb2885f2759eecae2870d95030c601d41"


def get_response_time_percentile(content, percentile=99.0):
    min_time = 0.0
    histograms = content.split("\n\n")
    for histogram in histograms[1:]:
        percents = re.findall("h.*\(.*\.\.(\d*)\).*,.*\((.*)%\)", histogram)
        for timing, percent in percents:
            if timing:
                timing = float(timing)
                if float(percent) <= 100.0 - percentile:
                    min_time = timing if not min_time else min(min_time, timing)
                    break
    return min_time


def check_port(port):
    s = socket.socket()
    address = 'localhost'
    logging.debug("Attempting to connect to %s on port %s" % (address, port))
    try:
        s.connect((address, port))
        logging.debug("Connected to %s on port %s" % (address, port))
        return True
    except socket.error, e:
        logging.debug("Connection to %s on port %s failed: %s" % (address, port, e))
        return False
    finally:
        s.close()


def get_shell_command_output(command):
    return sp.Popen(command, shell=True, stdout=sp.PIPE, stderr=sp.PIPE, cwd=os.getcwd()).communicate()[0]


def generate_shoot_info():
    dump_stat = get_shell_command_output(RUN_DOLBILKA_DUMPER_CMD)
    percentiles = ""

    for percentile in [50, 75, 90, 95, 99, 99.9]:
        time_percentile = get_response_time_percentile(dump_stat, percentile)
        percentiles += "Percentile %f of response time (microseconds): %f\n" % (percentile, time_percentile)

    requests_by_seconds = get_shell_command_output("./d-dumper -T -f dump.bin | awk '{print $2}' | grep -Eo '^.{10}' | sort | uniq -c")
    responses_by_seconds = get_shell_command_output("./d-dumper -T -f dump.bin | awk '{print $3}' | grep -Eo '^.{10}' | sort | uniq -c")

    return percentiles + "\n" + dump_stat + "\n" + "Requests by seconds:\n" + requests_by_seconds + "\nResponses by seconds:\n" + responses_by_seconds


class AntirobotLoadTestingTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        dolbilka_rps = sdk2.parameters.Integer(
            'Dolbilka RPS',
            default=DOLBILKA_RPS,
            required=False,
        )

        dolbilka_requests_limit = sdk2.parameters.Integer(
            'Dolbilka requests limit',
            default=DOLBILKA_REQUESTS_LIMIT,
            required=False,
        )

        dolbilka_additional_parameters = sdk2.parameters.String(
            'Dolbilka additional parameters',
            default="",
            required=False,
        )

        antirobot_bundle = sdk2.parameters.Resource(
            'Antirobot binaries',
            resource_type=ANTIROBOT_BUNDLE,
            required=True,
        )

        dolbilka_rbtorrent = sdk2.parameters.String(
            'Dolbilka binaries rbtorrent',
            default=DOLBILKA_BINARIES_RBTORRENT,
            required=True,
        )

        antirobot_data_rbtorrent = sdk2.parameters.String(
            'Antirobot data rbtorrent',
            default=ANTIROBOT_DATA_RBTORRENT,
            required=True,
        )

        antirobot_formulas_rbtorrent = sdk2.parameters.String(
            'Antirobot formulas rbtorrent',
            default=ANTIROBOT_FORMULAS_RBTORRENT,
            required=True,
        )

        mocks_rbtorrent = sdk2.parameters.String(
            'Captcha and wizard mocks rbtorrent',
            default=MOCKS_RBTORRENT,
            required=True,
        )

        plan_rbtorrent = sdk2.parameters.String(
            'Dolbilka plan rbtorrent',
            default=DOLBILKA_PLAN_RBTORRENT,
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2660V1 & ctc.Tag.HDD

    def get_dolbilka_execute_cmd(self):
        parameters = {
            'request_limit': self.Parameters.dolbilka_requests_limit,
            'rps': self.Parameters.dolbilka_rps,
            'shoot_port': ANTIROBOT_CACHER_PORT
        }
        return RUN_DOLBILKA_EXECUTE_CMD.format(**parameters) + " " + str(self.Parameters.dolbilka_additional_parameters)

    def execute_shell_command(self, command, description, wait=False, port=None):
        with ProcessLog(self, logger=logging.getLogger(description)) as process_log:
            popen = sp.Popen(command, shell=True, stdout=process_log.stdout, stderr=process_log.stdout, cwd=os.getcwd())

            if not wait:
                return

            if port is not None:
                while popen.returncode is None and not check_port(port):
                    time.sleep(5)
                    popen.poll()
                return popen.returncode

            return popen.wait()

    def generate_config(self):
        parameters = {
            'cacher_port': ANTIROBOT_CACHER_PORT,
            'process_port': ANTIROBOT_PROCESSOR_PORT,
            'admin_port': ANTIROBOT_ADMIN_PORT,
            'base_dir': os.getcwd().replace('/', '\\/'),
            'wizard_port': WIZARD_MOCK_PORT,
            'captcha_port': CAPTCHA_MOCK_PORT,
        }
        command = ("./antirobot_gencfg --geo=man --active-instances=production_antirobot_iss "
                   "--port={cacher_port} --admin-port={admin_port} --process-port={process_port} | "
                   "sed -e 's/BaseDir = .*$/BaseDir = {base_dir}/' | "
                   "sed -e 's/AllDaemons = .*$/AllDaemons = localhost:{admin_port};[::1]:{admin_port}/' | "
                   "sed -e 's/RemoteWizards .*$/RemoteWizards = localhost:{wizard_port};[::1]:{wizard_port}/' | "
                   "sed -e 's/Page404Host .*$/Page404Host = localhost;[::]/' | "
                   "sed -e 's/CaptchaApiHost .*$/CaptchaApiHost = localhost:{captcha_port};[::1]:{captcha_port}/'"
                   " > antirobot.lua").format(**parameters)
        self.execute_shell_command(command, "gencfg", wait=True)

    def prepare_external_data(self):
        antirobot_binary_tar_path = str(sdk2.ResourceData(self.Parameters.antirobot_bundle).path)
        self.execute_shell_command("tar xvf '%s'" % antirobot_binary_tar_path, "tar", wait=True)

        copy.RemoteCopy(self.Parameters.dolbilka_rbtorrent, ".")()
        self.execute_shell_command("tar xvf resource.tar.gz", "tar", wait=True)

        copy.RemoteCopy(self.Parameters.mocks_rbtorrent, ".")()
        self.execute_shell_command("tar xvf resource.tar.gz", "tar", wait=True)

        copy.RemoteCopy(self.Parameters.antirobot_data_rbtorrent, ".")()
        copy.RemoteCopy(self.Parameters.antirobot_formulas_rbtorrent, ".")()

        copy.RemoteCopy(self.Parameters.plan_rbtorrent, ".")()

        os.mkdir("logs")

    def store_results(self):
        results = [
            "dump.bin",
            "antirobot.lua",
            "logs",
        ]
        results_folder = "results"

        parameters = {
            'results': " ".join(results),
            'results_folder': results_folder,
        }

        os.mkdir(results_folder)
        self.execute_shell_command("cp -r {results} {results_folder}".format(**parameters), "copy results", wait=True)

    def on_execute(self):
        self.prepare_external_data()

        self.generate_config()

        self.execute_shell_command(RUN_CAPTCHA_MOCK_CMD, "captcha_mock", wait=True, port=CAPTCHA_MOCK_PORT)
        self.execute_shell_command(RUN_WIZARD_MOCK_CMD, "wizard_mock", wait=True, port=WIZARD_MOCK_PORT)

        self.execute_shell_command(RUN_ANTIROBOT_CMD, "antirobot", wait=True, port=ANTIROBOT_CACHER_PORT)

        self.execute_shell_command(self.get_dolbilka_execute_cmd(), "dolbilka", wait=True)

        self.Context.shoot_info = generate_shoot_info()

        self.store_results()

        resource = sdk2.ResourceData(AntirobotLoadTestingResource(self, "Load testing artifacts", "results"))
        resource.ready()

    @sdk2.footer()
    def footer(self):
        return "<pre>" + self.Context.shoot_info + "</pre>"
