# -*- coding: utf-8 -*-

import json

from sandbox.projects.antirobot.LoadTestingV3 import AntirobotTankLoadTestingTask
from sandbox import sdk2
from sandbox import common
import sandbox.common.types.task as ctt
from sandbox.projects.resource_types import ANTIROBOT_BUNDLE

fields = (
    'rps_real',
    'p50_response_time_ms',
    'p75_response_time_ms',
    'p80_response_time_ms',
    'p85_response_time_ms',
    'p90_response_time_ms',
    'p95_response_time_ms',
    'p98_response_time_ms',
    'p99_response_time_ms',
    'p999_response_time_ms',
    'p9999_response_time_ms',
    'max_response_time_ms',
    'rps_http_server_nonfailed',
    'spilled_forwarding_rps',
    'spilled_processing_rps',
    'rps_process_server',
    'memory_rss_bytes',
)


class AntirobotTankLoadTestingTaskChooseOne(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        antirobot_bundle = sdk2.parameters.Resource(
            'Antirobot binaries',
            resource_type=ANTIROBOT_BUNDLE,
            required=True,
        )

        subtasks_number = sdk2.parameters.Integer(
            'Number of subtasks',
            default=3,
            required=True,
        )

        with sdk2.parameters.String("Choose one by", required=True) as chose_one_by:
            for field in fields:
                chose_one_by.values[field] = chose_one_by.Value(field)

            chose_one_by.values['rps_real'] = chose_one_by.Value('rps_real', default=True)

    def _create_subtasks(self):
        subtasks = []
        for i in xrange(int(self.Parameters.subtasks_number)):
            subtask = AntirobotTankLoadTestingTask(
                self,
                owner=self.owner,
                description="Child of AntirobotTankLoadTestingTaskMedian task {}".format(self.id),
                priority=self.Parameters.priority,
                antirobot_bundle=self.Parameters.antirobot_bundle,
            )
            subtasks.append(subtask.enqueue().id)
        return subtasks

    def on_execute(self):
        with self.memoize_stage.create_children:
            subtasks = self._create_subtasks()
            self.Context.subtask_ids = subtasks

            raise sdk2.WaitTask(subtasks, set(common.utils.chain(ctt.Status.Group.FINISH, ctt.Status.Group.BREAK)),
                                wait_all=True)
        subtasks = list(self.find(id=self.Context.subtask_ids).limit(int(self.Parameters.subtasks_number)))

        failed = [subtask for subtask in subtasks if subtask.status not in ctt.Status.Group.SUCCEED]
        if failed:
            raise common.errors.TaskFailure("Failed subtasks: {}".format(failed))

        self.Context.subtasks = []
        for subtask in subtasks:
            d = dict()
            for field in fields:
                d[field] = json.loads(subtask.Context.testenv_values)[field]
            self.Context.subtasks.append((subtask.id, d))

        choose_by = self.Parameters.chose_one_by
        arr = [(json.loads(subtask.Context.testenv_values)[choose_by], i) for i, subtask in enumerate(subtasks)]

        median_idx = sorted(arr)[len(arr) / 2][1]
        median_subtask = subtasks[median_idx]

        testenv_values = dict()
        for field in fields:
            testenv_values[field] = json.loads(median_subtask.Context.testenv_values)[field]

        self.Context.testenv_values = json.dumps(testenv_values)

    @sdk2.header()
    def header(self):
        if self.Context.subtasks:
            html = "<table class=\"cf__t\"><thead><tr><td>ID</td>"
            for field in fields:
                html += "<td>%s</td>" % field
            html += "</tr></thead>"
            html += "<tbody>"
            for subtask_id, d in self.Context.subtasks:
                html += "<tr>"
                url = "https://sandbox.yandex-team.ru/task/%d" % subtask_id
                html += "<td><a href=\"%s\">%d</a></td>" % (url, subtask_id)
                for field in fields:
                    if field in d:
                        html += "<td>%.3f</td>" % float(d[field])
                    else:
                        html += "<td>NODATA</td>"
                html += "</tr>"
            html += "</tbody></table>"
            return html
        else:
            return 'Waiting for subtasks...'
