import logging
import os
import socket
import time

from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.search import performance as search_performance
from sandbox.projects.common.search.components import StandardSearch
from sandbox.sdk2.helpers import ProcessLog

from sandbox import sdk2
from sandbox.sandboxsdk import copy
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc
from sandbox.projects.resource_types import ANTIROBOT_BUNDLE

CAPTCHA_MOCK_PORT = 10808
WIZARD_MOCK_PORT = 8891
ANTIROBOT_CACHER_PORT = 8899
ANTIROBOT_PROCESSOR_PORT = 8898
ANTIROBOT_ADMIN_PORT = 8897
DOLBILKA_RPS = 1000
DOLBILKA_REQUESTS_LIMIT = 60000

RUN_ANTIROBOT_CMD = "./antirobot_daemon -c antirobot.lua --no-tvm"
RUN_CAPTCHA_MOCK_CMD = "./api_captcha_mock -p %d" % CAPTCHA_MOCK_PORT
RUN_WIZARD_MOCK_CMD = "./wizard_mock %d" % WIZARD_MOCK_PORT

MOCKS_RBTORRENT = "rbtorrent:9f32960452c5fedd6d8c051d99eccf4a08089ec9"
ANTIROBOT_FORMULAS_RBTORRENT = "rbtorrent:870a541ba9475eb971f9698f74d3680dd256ed39"
ANTIROBOT_DATA_RBTORRENT = "rbtorrent:13994fbda2b7566f481c4c978ee407bad2ab741c"


def check_port(port):
    s = socket.socket()
    address = 'localhost'
    logging.debug("Attempting to connect to %s on port %s" % (address, port))
    try:
        s.connect((address, port))
        logging.debug("Connected to %s on port %s" % (address, port))
        return True
    except socket.error, e:
        logging.debug("Connection to %s on port %s failed: %s" % (address, port, e))
        return False
    finally:
        s.close()


class ConfigStub:
    def apply_local_patch(selfs, **kwargs):
        pass

    def save_to_file(self, one):
        pass


class ConfigClassStub:
    def get_config_from_file(self, one):
        return ConfigStub()

    def get_production_config(self, **kwargs):
        return ConfigStub()


class Antirobot(StandardSearch):
    name = 'antirobot'

    def __init__(
            self,
            work_dir,
            binary,
            port,
            config_class,
            config_file,
            data_dir,
            runtime_dir=None,
            config_params='',
            use_profiler=False,
            task=None,
            **kwargs
    ):
        super(Antirobot, self).__init__(
            work_dir,
            binary,
            port,
            config_class,
            config_file,
            config_params,
            use_profiler=use_profiler,
            use_gperftools=use_profiler,  # if using profiler, use gperf tools' one
            task=task,
            **kwargs
        )
        self.data_dir = data_dir
        self.runtime_dir = runtime_dir
        self.prepared_data_dir = self.prepare_data(self.data_dir, self.runtime_dir)
        self.config_file = config_file

    def _get_run_cmd(self, config_path):
        cmd = [
            '-c', "antirobot.lua",
            '-p', str(self.port),
            '--no-tvm',
        ]
        return cmd

    @property
    def _stop_command(self):
        return "/admin?action=shutdown"

    def prepare_data(self, data_dir, runtime_dir):
        return "."


class AntirobotTestPerformance(
    search_performance.OldShootingTask,
    sdk2.Task
):
    class Parameters(sdk2.Task.Parameters):
        dolbilka_rps = sdk2.parameters.Integer(
            'Dolbilka RPS',
            default=DOLBILKA_RPS,
            required=False,
        )

        dolbilka_requests_limit = sdk2.parameters.Integer(
            'Dolbilka requests limit',
            default=DOLBILKA_REQUESTS_LIMIT,
            required=False,
        )

        dolbilka_time_limit = sdk2.parameters.Integer(
            'Dolbilka time limit (seconds)',
            default=60,
            required=False,
        )

        dolbilka_sessions = sdk2.parameters.Integer(
            'Dolbilka sessions count',
            default=10,
            required=False,
        )

        dolbilka_max_simultaneous_requests = sdk2.parameters.Integer(
            'Dolbilka maximum simultaneous requests',
            default=26,
            required=False,
        )

        dolbilka_mode = sdk2.parameters.String(
            'Dolbilka executor mode',
            default="plan",
            required=False,
        )

        dolbilka_plan_resource_id = sdk2.parameters.Integer(
            'Dolbilka plan resource id',
            default=746939454,
            required=False,
        )

        antirobot_bundle = sdk2.parameters.Resource(
            'Antirobot binaries',
            resource_type=ANTIROBOT_BUNDLE,
            required=True,
        )

        antirobot_data_rbtorrent = sdk2.parameters.String(
            'Antirobot data rbtorrent',
            default=ANTIROBOT_DATA_RBTORRENT,
            required=True,
        )

        antirobot_formulas_rbtorrent = sdk2.parameters.String(
            'Antirobot formulas rbtorrent',
            default=ANTIROBOT_FORMULAS_RBTORRENT,
            required=True,
        )

        mocks_rbtorrent = sdk2.parameters.String(
            'Captcha and wizard mocks rbtorrent',
            default=MOCKS_RBTORRENT,
            required=True,
        )

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2660V1 & ctc.Tag.HDD

    class ContextWrapper:
        def __init__(self, task):
            self.Task = task

        def get(self, key, default=None):
            if key not in self.Task.Context:
                return default
            return self[key]

        def set(self, key, value):
            self[key] = value

        def setdefault(self, key, default=None):
            if key not in self.Task.Context:
                self.set(key, default)
            return self.get(key)

        def __getitem__(self, key):
            if key not in self.Task.Context:
                return None
            return getattr(self.Task.Context, key)

        def __setitem__(self, key, value):
            setattr(self.Task.Context, key, value)

        def __iter__(self):
            for s in self.Task.Context:
                yield s[0]

    descr = "Antirobot task"

    def sync_resource(self, id):
        return channel.task.sync_resource(id)

    def abs_path(self, x=None):
        return channel.task.abs_path(x)

    def create_resource(self, description, work_dir, resource_type, **kwargs):
        x = channel.task.create_resource(description, work_dir, resource_type, **kwargs)
        x.path = str(x.path)
        return x

    def execute_shell_command(self, command, description, port=None):
        with ProcessLog(self, logger=logging.getLogger(description)) as process_log:
            popen = sp.Popen(command, shell=True, stdout=process_log.stdout, stderr=process_log.stdout, cwd=os.getcwd())

            if port is not None:
                while popen.returncode is None and not check_port(port):
                    time.sleep(5)
                    popen.poll()
                return popen.returncode

            return popen.wait()

    def generate_config(self):
        parameters = {
            'cacher_port': ANTIROBOT_CACHER_PORT,
            'process_port': ANTIROBOT_PROCESSOR_PORT,
            'admin_port': ANTIROBOT_ADMIN_PORT,
            'base_dir': os.getcwd().replace('/', '\\/'),
            'wizard_port': WIZARD_MOCK_PORT,
            'captcha_port': CAPTCHA_MOCK_PORT,
        }
        command = ("./antirobot_gencfg --geo=man --active-instances=production_antirobot_iss "
                   "--port={cacher_port} --admin-port={admin_port} --process-port={process_port} | "
                   "sed -e 's/BaseDir = .*$/BaseDir = {base_dir}/' | "
                   "sed -e 's/AllDaemons = .*$/AllDaemons = localhost:{admin_port};[::1]:{admin_port}/' | "
                   "sed -e 's/RemoteWizards .*$/RemoteWizards = localhost:{wizard_port};[::1]:{wizard_port}/' | "
                   "sed -e 's/Page404Host .*$/Page404Host = localhost;[::]/' | "
                   "sed -e 's/CaptchaApiHost .*$/CaptchaApiHost = localhost:{captcha_port};[::1]:{captcha_port}/'"
                   " > antirobot.lua").format(**parameters)
        self.execute_shell_command(command, "gencfg")

    def prepare_external_data(self):
        antirobot_binary_tar_path = str(sdk2.ResourceData(self.Parameters.antirobot_bundle).path)
        self.execute_shell_command("tar xvf '%s'" % antirobot_binary_tar_path, "tar")

        copy.RemoteCopy(self.Parameters.mocks_rbtorrent, ".")()
        self.execute_shell_command("tar xvf resource.tar.gz", "tar")

        copy.RemoteCopy(self.Parameters.antirobot_data_rbtorrent, ".")()
        copy.RemoteCopy(self.Parameters.antirobot_formulas_rbtorrent, ".")()

        os.mkdir("logs")

    def on_enqueue(self):
        sdk2.Task.on_enqueue(self)

    def on_execute(self):
        self.ctx = self.ContextWrapper(self)
        self.prepare_external_data()

        self.generate_config()

        self.execute_shell_command(RUN_CAPTCHA_MOCK_CMD, "captcha_mock", port=CAPTCHA_MOCK_PORT)
        self.execute_shell_command(RUN_WIZARD_MOCK_CMD, "wizard_mock", port=WIZARD_MOCK_PORT)

        wizard = Antirobot(
            work_dir=channel.task.abs_path(),
            binary="antirobot_daemon",
            port=ANTIROBOT_CACHER_PORT,
            config_class=ConfigClassStub(),
            config_file="antirobot.lua",
            data_dir="data",
        )

        self._init_virtualenv()
        self.ctx["dolbilka_executor_sessions"] = self.Parameters.dolbilka_sessions
        self.ctx["dolbilka_fixed_rps"] = self.Parameters.dolbilka_rps
        self.ctx["dolbilka_executor_max_simultaneous_requests"] = self.Parameters.dolbilka_max_simultaneous_requests
        self.ctx["dolbilka_executor_mode"] = self.Parameters.dolbilka_mode
        self.ctx["dolbilka_executor_requests_limit"] = self.Parameters.dolbilka_requests_limit
        self.ctx["dolbilka_executor_time_limit"] = self.Parameters.dolbilka_time_limit
        with wizard:
            self._old_shoot(wizard, self.Parameters.dolbilka_plan_resource_id)

    def get_short_task_result(self):
        if self.is_completed() and "max_rps" in self.ctx:
            return "{:0.2f}".format(self.ctx["max_rps"])

    @property
    def footer(self):
        self.ctx = self.ContextWrapper(self)
        return search_performance.OldShootingTask.footer.fget(self)
