# -*- coding: utf-8 -*-

import logging
import os
import socket
import time
import urllib2
import psutil
import json
import re
import shutil
import getpass

from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc
from sandbox.projects.resource_types import ANTIROBOT_BUNDLE, ANTIROBOT_FORMULAS, ANTIROBOT_DATA
from sandbox.projects.common import file_utils
from sandbox import common
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.tank.load_resources.resources import YANDEX_TANK_VIRTUALENV_19, PHANTOM_EXECUTABLE
from sandbox.projects.antirobot.MergeJsonConfigFilesTask import AntirobotJsonConfigFileResource
from sandbox.projects.antirobot.UpdateServiceIdentifier import AntirobotServiceIdentifierJsonFileResource

from sandbox.projects.common.nanny import client as nanny_client

MOCK_BINARIES_DEFAULT_RESOURCE = 1636946817
PHANTOM_AMMO_DEFAULT_RESOURCE = 766466859
TANK_VIRTUALENV_DEFAULT_RESOURCE = 381538594
PHANTOM_BINARY_DEFAULT_RESOURCE = 99315604
TANK_PATCH_DEFAULT_RESOURCE = 792798232
UNISTAT_PROXY_DEFAULT_RESOURCE = 790826278

CAPTCHA_MOCK_PORT = 10808
FURY_MOCK_PORT = 10819
WIZARD_MOCK_PORT = 8891
ANTIROBOT_CACHER_PORT = 8899
ANTIROBOT_PROCESSOR_PORT = 8898
ANTIROBOT_ADMIN_PORT = 8897
UNISTAT_PROXY_PORT = 9797
ANTIROBOT_CACHER_PORT2 = 8900
ANTIROBOT_PROCESSOR_PORT2 = 8901
ANTIROBOT_ADMIN_PORT2 = 8902

RUN_PROCESSOR_MOCK_CMD = "./processor_mock --port %d" % ANTIROBOT_PROCESSOR_PORT
RUN_UNISTAT_PROXY_CMD = "./unistat_proxy --http-port %d --unistat-port %d" % (UNISTAT_PROXY_PORT, ANTIROBOT_ADMIN_PORT)

QUANTILES = ["0", "50", "75", "85", "90", "95", "97", "99", "999", "9999"]

PROCESSOR_ON_CACHER = 'ON_CACHER'
PROCESSOR_MOCK = 'MOCK'
PROCESSOR_PORT_NOT_EXISTS = 'NOT_EXISTS'

NON_EXISTENT_PORT = 9999


class AntirobotTankLoadTestingLogsResource(sdk2.Resource):
    """
    Antirobot load testing logs
    """
    any_arch = True
    executable = False
    releasable = False


class AntirobotTankLoadTestingDataResource(sdk2.Resource):
    """
    Antirobot load testing data
    """
    any_arch = True
    executable = False
    releasable = False


class AntirobotTankLoadTestingPerfDataResource(sdk2.Resource):
    """
    Antirobot load testing data
    """
    any_arch = True
    executable = False
    releasable = False


def check_port(port):
    s = socket.socket()
    address = 'localhost'
    logging.debug("Attempting to connect to %s on port %s" % (address, port))
    try:
        s.connect((address, port))
        logging.debug("Connected to %s on port %s" % (address, port))
        return True
    except socket.error, e:
        logging.debug("Connection to %s on port %s failed: %s" % (address, port, e))
        return False
    finally:
        s.close()


def get_all_antirobot_processes():
    processes = [proc for proc in psutil.process_iter()
                 if proc.name == "antirobot_daemon" and proc.username == getpass.getuser()]

    if len(processes) == 0:
        logging.warn("No process with name 'antirobot_daemon'")

    if len(processes) > 1:
        logging.warn("More than one process with name 'antirobot_daemon'")

    return processes


def shutdown_antirobot(admin_port):
    try:
        urllib2.urlopen('http://localhost:{}/admin?action=shutdown'.format(admin_port)).read()
        time.sleep(20)
    except Exception as e:
        logging.warn(e)

    processes = get_all_antirobot_processes()

    for process in processes:
        if process.is_running():
            process.terminate()
            time.sleep(5)
            if process.is_running():
                process.kill()


def get_shell_command_output(command):
    return sp.Popen(command, shell=True, stdout=sp.PIPE, stderr=sp.PIPE, cwd=os.getcwd()).communicate()[0]


def to_response_metric_name(percent_as_string):
    if percent_as_string.startswith("100"):
        return "max_response_time_ms"

    if '.' not in percent_as_string:
        percentile = percent_as_string
    else:
        percentile = percent_as_string.rstrip('0').replace('.', '')

    return "p{}_response_time_ms".format(percentile)


def find_all(text, regexp):
    r = re.compile(regexp)
    return r.findall(text)


def find_first(text, regexp):
    return find_all(text, regexp)[0]


def get_signal_value(signal_name, alternative_signal_name=None):
    output = get_shell_command_output("curl \"localhost:{}/?metric={}\"".format(UNISTAT_PROXY_PORT, signal_name))

    if alternative_signal_name is None or 'Empty metric' not in output:
        return float(output)

    command = "curl \"localhost:{}/?metric={}\"".format(UNISTAT_PROXY_PORT, alternative_signal_name)
    alternative_signal_output = get_shell_command_output(command)
    return float(alternative_signal_output)


def parse_tank_ui_output(text):
    percent_re_string = r'(\d{1,3}(?:\.\d+)?)%'
    percentiles = find_all(text, percent_re_string + r' < (\d+(?:\.\d+)?) ms')
    d = dict()
    for percent, count in percentiles:
        d[to_response_metric_name(percent)] = float(count)
    try:
        d['fail_percent'] = float(find_first(text, percent_re_string + ': 0 N/A'))
    except:
        d['fail_percent'] = 0.0
    duration_string = find_first(text, r'Duration: (\d{1,2}:\d\d:\d\d)')
    duration_hours, duration_minutes, duration_seconds = duration_string.split(':', 3)
    d['duration_seconds'] = (int(duration_hours)*60 + int(duration_minutes))*60 + int(duration_seconds)
    d['spilled_forwarding_rps'] = get_signal_value('forwarding_queue.forwarding_reqs.spilled_reqs_deee',
                                                   'processor_response_apply_queue.spilled_applies_deee') / d['duration_seconds']
    d['spilled_processing_rps'] = get_signal_value('processing_queue.processing_reqs.spilled_reqs_deee') / d['duration_seconds']
    d['rps_http_server_nonfailed'] = get_signal_value('http_server.time_10s_deee') * ((100.0 - d['fail_percent']) / 100.0) / d['duration_seconds']
    d['rps_real'] = d['rps_http_server_nonfailed'] - d['spilled_forwarding_rps'] - d['spilled_processing_rps']
    d['rps_process_server'] = get_signal_value('process_server.time_10s_deee') / d['duration_seconds']

    return d


class AntirobotTankLoadTestingTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        antirobot_bundle = sdk2.parameters.Resource(
            'Antirobot binaries',
            resource_type=ANTIROBOT_BUNDLE,
            required=True,
        )

        antirobot_data_archive = sdk2.parameters.LastReleasedResource(
            'Antirobot data archive',
            resource_type=ANTIROBOT_DATA,
            required=True,
        )

        antirobot_formulas_archive = sdk2.parameters.LastReleasedResource(
            'Antirobot formulas archive',
            resource_type=ANTIROBOT_FORMULAS,
            required=True,
        )

        mock_binaries = sdk2.parameters.Resource(
            'Captcha and wizard mocks binaries',
            default_value=MOCK_BINARIES_DEFAULT_RESOURCE,
            required=True,
        )

        ammo_file = sdk2.parameters.Resource(
            'Phantom ammo file',
            default_value=PHANTOM_AMMO_DEFAULT_RESOURCE,
            required=True,
        )

        phantom_schedule = sdk2.parameters.String(
            'Phantom shooting schedule',
            default="line(1, 30000, 1m)",
            required=True,
        )

        cfs_period_us = sdk2.parameters.Integer(
            'cgroup cpu.cfs_period_us',
            default=100000,
            required=True,
        )

        cpu_cores_limit = sdk2.parameters.Float(
            'Limit for CPU cores',
            default=26.3363,
            required=True,
        )

        phantom_instances = sdk2.parameters.Integer(
            'Number of simultaneous connections from Phantom',
            default=200,
            required=True,
        )

        phantom_timeout = sdk2.parameters.String(
            'Phantom response timeout',
            default="0.1s",
            required=True,
        )

        make_flamegraph = sdk2.parameters.Bool(
            'Make flamegraph (requires -fno-omit-frame-pointer)',
            default=False,
            required=True,
        )

        tank_virtualenv = sdk2.parameters.Resource(
            'Tank virtualenv archive',
            resource_type=YANDEX_TANK_VIRTUALENV_19,
            default_value=TANK_VIRTUALENV_DEFAULT_RESOURCE,
            required=True,
        )

        phantom_binary = sdk2.parameters.Resource(
            'Phantom binary',
            resource_type=PHANTOM_EXECUTABLE,
            default_value=PHANTOM_BINARY_DEFAULT_RESOURCE,
            required=True,
        )

        tank_patch = sdk2.parameters.Resource(
            'Tank patch',
            default_value=TANK_PATCH_DEFAULT_RESOURCE,
            required=True,
        )

        unistat_proxy_binary = sdk2.parameters.Resource(
            'Unistat proxy binary',
            default_value=UNISTAT_PROXY_DEFAULT_RESOURCE,
            required=True,
        )

        loop = sdk2.parameters.Integer(
            "Don't stop when the end of ammo is reached but loop it N times",
            default=1,
        )

        with sdk2.parameters.String("Processor behaviour", required=True) as processor_behaviour:
            processor_behaviour.values[PROCESSOR_ON_CACHER] = processor_behaviour.Value("Same as cacher", default=True)
            processor_behaviour.values[PROCESSOR_MOCK] = processor_behaviour.Value("Use mock")

            # Like cacher, but without network overhead and with exceptions
            processor_behaviour.values[PROCESSOR_PORT_NOT_EXISTS] = processor_behaviour.Value("Not existing port")

        add_sanitizer_options = sdk2.parameters.Bool(
            'Add ASan and MemSan options (symbolize=1:allow_addr2line=1)',
            default=False,
            required=True,
        )

        api_captcha_mock_timeout_microseconds = sdk2.parameters.Integer(
            'Timeout for api_captcha_mock to wait before response (in microseconds)',
            default=10000,
            required=True,
        )

        disable_fury_mock = sdk2.parameters.Bool(
            "Disable fury mock (use service from config)",
            default=False,
        )

        with disable_fury_mock.value[False]:
            fury_mock_timeout_microseconds = sdk2.parameters.Integer(
                'Timeout for fury_mock to wait before response (in microseconds)',
                default=70000,
                required=True,
            )

        use_tvm = sdk2.parameters.Bool(
            "Use TVM",
            default=False,
        )

        with use_tvm.value[True]:
            tvm_secret_vault_owner = sdk2.parameters.String(
                'Vault owner of TVM secret',
                default_value=''
            )

            tvm_secret_vault_name = sdk2.parameters.String(
                'Vault item containing TVM secret',
                required=True,
                default_value='tvm-secret'
            )

        prod_antirobot = sdk2.parameters.Bool(
            "Run with production release (compatibility test)",
            default=False,
        )

        with prod_antirobot.value[True]:
            nanny_oauth_token_vault_owner = sdk2.parameters.String(
                'Vault owner of Nanny OAuth token',
                default_value=''
            )

            nanny_oauth_token_vault_name = sdk2.parameters.String(
                'Vault item containing Nanny OAuth token',
                required=True,
                default_value='nanny-oauth-token'
            )

            antirobot_prod_bundle = sdk2.parameters.Resource(
                'Antirobot prod binaries (keep empty by default)',
                resource_type=ANTIROBOT_BUNDLE,
            )

            antirobot_prod_formulas_archive = sdk2.parameters.Resource(
                'Antirobot prod formulas archive (keep empty by default)',
                resource_type=ANTIROBOT_FORMULAS,
            )

            antirobot_prod_data_archive = sdk2.parameters.Resource(
                'Antirobot prod data archive (keep empty by default)',
                resource_type=ANTIROBOT_DATA,
            )

            service_config_prod_json = sdk2.parameters.Resource(
                'service_config.json prod (keep empty by default)',
                resource_type=AntirobotJsonConfigFileResource,
            )

            service_identifier_prod_json = sdk2.parameters.Resource(
                'service_identifier.json prod (keep empty by default)',
                resource_type=AntirobotServiceIdentifierJsonFileResource,
            )

            shoot_to_prod = sdk2.parameters.Bool(
                "Tank will send requests to prod version (True) or to current version (False)",
                default=False,
            )

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL & ctc.Tag.INTEL_E5_2660V1 & ctc.Tag.HDD

    def execute_shell_command(self, command, description, wait=False, port=None, callback=None):
        with ProcessLog(self, logger=logging.getLogger(description)) as process_log:
            popen = sp.Popen(command, shell=True, stdout=process_log.stdout, stderr=process_log.stdout, cwd=os.getcwd())

            if not wait:
                return

            if port is not None:
                while popen.returncode is None and not check_port(port):
                    time.sleep(1)
                    popen.poll()
                return popen.returncode

            while True:
                try:
                    return popen.wait(1)
                except sp.TimeoutExpired:
                    if callback:
                        callback()

    def gen_processors_cfg(self, n, ports):
        result = []
        for i in range(n):
            result.append("localhost:{0};[::1]:{0}".format(ports[i % len(ports)]))
        return " ".join(result)

    def generate_config(self, bin_dir, cacher_port, admin_port, processor_port, processor_ports):
        parameters = {
            'bin_dir': bin_dir,
            'cacher_port': cacher_port,
            'admin_port': admin_port,
            'base_dir': os.path.join(os.getcwd(), bin_dir).replace('/', '\\/'),
            'wizard_port': WIZARD_MOCK_PORT,
            'captcha_port': CAPTCHA_MOCK_PORT,
            'fury_port': FURY_MOCK_PORT,
        }

        if self.Parameters.processor_behaviour == PROCESSOR_ON_CACHER:
            parameters['process_port'] = processor_port
            parameters['all_daemons'] = self.gen_processors_cfg(200, processor_ports)
        elif self.Parameters.processor_behaviour == PROCESSOR_MOCK:
            parameters['process_port'] = NON_EXISTENT_PORT
            parameters['all_daemons'] = self.gen_processors_cfg(200, processor_ports)
        elif self.Parameters.processor_behaviour == PROCESSOR_PORT_NOT_EXISTS:
            parameters['process_port'] = processor_port
            parameters['all_daemons'] = self.gen_processors_cfg(200, [NON_EXISTENT_PORT])

        command = ("./{bin_dir}/antirobot_gencfg --local " +
                   "--port={cacher_port} --admin-port={admin_port} --process-port={process_port} | " +
                   "sed -e 's/BaseDir = .*$/BaseDir = {base_dir}/' | " +
                   "sed -e 's/AllDaemons = .*$/AllDaemons = {all_daemons}/' | " +
                   "sed -e 's/RemoteWizards = .*$/RemoteWizards = localhost:{wizard_port};[::1]:{wizard_port}/' | " +
                   "sed -e 's/Page404Host = .*$/Page404Host = localhost;[::]/' | " +
                   "sed -e 's/FuryEnabled = .*$/FuryEnabled = 1/' | " +
                   ("" if self.Parameters.disable_fury_mock else "sed -e 's/FuryHost = .*$/FuryHost = localhost:{fury_port};[::1]:{fury_port}/' | ") +
                   "sed -e 's/CaptchaApiHost = .*$/CaptchaApiHost = localhost:{captcha_port};[::1]:{captcha_port}/'" +
                   " > {bin_dir}/data/antirobot.lua").format(**parameters)
        return_code = self.execute_shell_command(command, "gencfg", wait=True)
        assert return_code == 0, "antirobot_gencfg failed. See log for details."

    def get_telegraf_config_string(self):
        real_filename = os.path.join(os.path.dirname(os.path.realpath(__file__)), "metrics.txt")
        config = ""
        config += "<Monitoring>"
        config += "<Host address=\"[target]\" telegraf=\"./tank/bin/telegraf\">"
        for line in open(real_filename, 'r'):
            line = line.strip()
            if len(line) < 1:
                continue
            diff = line.endswith("deee")
            config += "<Custom measure=\"call\" "
            if diff:
                config += "diff=\"1\" "
                label = "customdiff-{}".format(line)
            else:
                label = line
            config += "label=\"{}\">".format(label)
            config += "curl \"localhost:{}/?metric={}\"".format(UNISTAT_PROXY_PORT, line)
            config += "</Custom>"
        config += "</Host>"
        config += "</Monitoring>"
        return config

    def get_task_url(self):
        return "https://sandbox.yandex-team.ru/task/%d/" % self.id

    def get_prod_resource(self, parameter, resource_type=None):
        overrided = getattr(self.Parameters, parameter)
        if overrided:
            return str(sdk2.ResourceData(self.Parameters.antirobot_prod_bundle).path)
        sandbox_files = self.runtime_attrs["content"]["resources"]["sandbox_files"]
        matched_files = []
        for file in sandbox_files:
            if resource_type is not None and file["resource_type"] == resource_type:
                matched_files.append(file)
        assert len(matched_files) == 1, "Wrong number of matched files for {}: {}".format(resource_type, json.dumps(matched_files, indent=4))
        file = matched_files[0]
        if "resource_id" in file:
            prod_resource = sdk2.Resource.find(id=file["resource_id"]).first()
            assert prod_resource, "Resource id={} not found".format(file["resource_id"])
        elif "task_id" in file:
            prod_resource = sdk2.Resource.find(task_id=file["task_id"]).first()
            assert prod_resource, "Resource task_id={} not found".format(file["task_id"])
        else:
            raise Exception("Neither id or task_id found in {}".format(json.dumps(file, indent=4)))
        return str(sdk2.ResourceData(prod_resource).path)

    def prepare_external_data(self, shoot_cacher_port):
        os.mkdir("antirobot")
        os.mkdir("antirobot_prod")
        os.mkdir("antirobot/logs")
        os.mkdir("antirobot_prod/logs")
        os.mkdir("logs")
        os.mkdir("logs/results")
        os.mkdir("tank")

        mock_binaries_tar_path = str(sdk2.ResourceData(self.Parameters.mock_binaries).path)
        self.execute_shell_command("tar xvf '%s'" % mock_binaries_tar_path, "tar", wait=True)
        antirobot_binary_tar_path = str(sdk2.ResourceData(self.Parameters.antirobot_bundle).path)
        self.execute_shell_command("tar xvf '%s' -C antirobot" % antirobot_binary_tar_path, "tar", wait=True)
        data_tar_path = str(sdk2.ResourceData(self.Parameters.antirobot_data_archive).path)
        self.execute_shell_command("tar xvf '%s' -C antirobot" % data_tar_path, "tar", wait=True)
        formulas_tar_path = str(sdk2.ResourceData(self.Parameters.antirobot_formulas_archive).path)
        self.execute_shell_command("tar xvf '%s' -C antirobot" % formulas_tar_path, "tar", wait=True)
        shutil.move("antirobot/arcadia_config/service_config.json", "antirobot/data/service_config.json")
        shutil.move("antirobot/arcadia_config/service_identifier.json", "antirobot/data/service_identifier.json")
        shutil.move("antirobot/arcadia_config/global_config.json", "antirobot/data/global_config.json")

        if self.Parameters.prod_antirobot:
            logging.debug("Production attrs:")
            logging.debug(json.dumps(self.runtime_attrs["content"]["resources"], indent=4))

            antirobot_binary_tar_path = self.get_prod_resource("antirobot_prod_bundle", resource_type="ANTIROBOT_BUNDLE")
            self.execute_shell_command("tar xvf '%s' -C antirobot_prod" % antirobot_binary_tar_path, "tar", wait=True)
            data_tar_path = self.get_prod_resource("antirobot_prod_data_archive", resource_type="ANTIROBOT_DATA")
            self.execute_shell_command("tar xvf '%s' -C antirobot_prod" % data_tar_path, "tar", wait=True)
            formulas_tar_path = self.get_prod_resource("antirobot_prod_formulas_archive", resource_type="ANTIROBOT_FORMULAS")
            self.execute_shell_command("tar xvf '%s' -C antirobot_prod" % formulas_tar_path, "tar", wait=True)
            shutil.copy(self.get_prod_resource("service_config_prod_json", resource_type="ANTIROBOT_JSON_CONFIG_FILE_RESOURCE"), "./antirobot_prod/data/")
            shutil.copy(self.get_prod_resource("service_identifier_prod_json", resource_type="ANTIROBOT_SERVICE_IDENTIFIER_JSON_FILE_RESOURCE"), "./antirobot_prod/data/")

        shutil.copy(str(sdk2.ResourceData(self.Parameters.ammo_file).path), "./ammo.txt")

        tank_virtualenv_tar_path = str(sdk2.ResourceData(self.Parameters.tank_virtualenv).path)
        self.execute_shell_command("tar xf '%s' --directory ./tank" % tank_virtualenv_tar_path, "tar", wait=True)

        shutil.copy(str(sdk2.ResourceData(self.Parameters.phantom_binary).path), "./tank/phantom_bin")

        shutil.copy(str(sdk2.ResourceData(self.Parameters.tank_patch).path), "./tank.patch")
        self.execute_shell_command("patch -p1 -i tank.patch", "patch", wait=True)

        shutil.copy(str(sdk2.ResourceData(self.Parameters.unistat_proxy_binary).path), "./unistat_proxy")

        logging.debug("Tank will shoot on %s port", shoot_cacher_port)
        file_utils.write_lines("load.ini", [
            "[phantom]",
            "address=localhost:" + str(shoot_cacher_port),
            "rps_schedule=" + str(self.Parameters.phantom_schedule),
            "phantom_path = ./tank/phantom_bin",
            "instances=" + str(self.Parameters.phantom_instances),
            "timeout=" + str(self.Parameters.phantom_timeout),
            "loop=" + str(self.Parameters.loop),
            "[tank]",
            "plugin_uploader=yandextank.plugins.DataUploader",
            "artifacts_dir=logs/results",
            "[meta]",
            "task=CAPTCHA-1363",
            "operator=robot-antirobot",
            "ignore_target_lock=1",
            "api_address=https://lunapark.yandex-team.ru/",
            "job_name=" + self.Parameters.description.replace('\n', '  '),
            "job_dsc=" + self.get_task_url(),
            "[telegraf]",
            "disguise_hostnames = 0",
            "config={}".format(self.get_telegraf_config_string()),
            "[console]",
            "disable_all_colors=1",
            "info_panel_width=50",
        ])

    def run_antirobot_daemon(self, bin_dir, cacher_port):
        cg = common.os.CGroup("antirobot_cgroup")
        cg.cpu["cfs_quota_us"] = str(int(self.Parameters.cpu_cores_limit * self.Parameters.cfs_period_us))
        cg.cpu["cfs_period_us"] = str(self.Parameters.cfs_period_us)

        cmd = "./{0}/antirobot_daemon -c {0}/data/antirobot.lua -L logs/".format(bin_dir)
        if not self.Parameters.use_tvm:
            cmd += " --no-tvm"

        if self.Parameters.prod_antirobot and self.Parameters.use_tvm:
            assert False, "Using tvm with production binary is forbidden for now"

        if self.Parameters.make_flamegraph:
            cmd = "perf record --call-graph fp -o perf.data " + cmd

        my_env = os.environ.copy()
        if self.Parameters.add_sanitizer_options:
            my_env["ASAN_OPTIONS"] = "symbolize=1:allow_addr2line=1"
            my_env["MSAN_OPTIONS"] = "symbolize=1:allow_addr2line=1"
        if self.Parameters.use_tvm:
            tvm_secret = sdk2.Vault.data(self.Parameters.tvm_secret_vault_owner or self.author,
                                         self.Parameters.tvm_secret_vault_name)
            my_env["ANTIROBOT_TVM_SECRET"] = tvm_secret

        err = open('logs/{0}.err'.format(bin_dir), 'w')
        popen = sp.Popen(cmd.split(' '), preexec_fn=cg.set_current, stderr=err, env=my_env)

        try_no = 0
        wait_seconds = 180
        while popen.returncode is None and not check_port(cacher_port):
            time.sleep(1)
            popen.poll()
            assert try_no < wait_seconds, "Antirobot is not started in {} seconds".format(wait_seconds)
            try_no += 1
        logging.info("{} return code is {}, port check {}".format(bin_dir, popen.returncode, check_port(cacher_port)))
        return popen.returncode

    def get_testenv_values(self):
        d = parse_tank_ui_output(self.Context.tank_ui_output)

        processes = get_all_antirobot_processes()

        for process in processes:
            if process.is_running():
                d['memory_rss_bytes'] = process.get_memory_info().rss
                break

        return json.dumps(d)

    def on_execute(self):
        if self.Parameters.prod_antirobot:
            nanny_oauth_token = sdk2.Vault.data(self.Parameters.nanny_oauth_token_vault_owner or self.author,
                                                self.Parameters.nanny_oauth_token_vault_name)
            self.nanny_client = nanny_client.NannyClient("https://nanny.yandex-team.ru", nanny_oauth_token)
            self.runtime_attrs = self.nanny_client.get_service_runtime_attrs(service_id="production_antirobot_iss")

        self.prepare_external_data(ANTIROBOT_CACHER_PORT2 if self.Parameters.shoot_to_prod else ANTIROBOT_CACHER_PORT)

        self.execute_shell_command("./api_captcha_mock -p %d --response-time-microseconds %d" % (CAPTCHA_MOCK_PORT, self.Parameters.api_captcha_mock_timeout_microseconds),
                                   "captcha_mock", wait=True, port=CAPTCHA_MOCK_PORT)
        self.execute_shell_command("./fury_mock -p %d --response-time-microseconds %d" % (FURY_MOCK_PORT, self.Parameters.fury_mock_timeout_microseconds),
                                   "fury_mock", wait=True, port=FURY_MOCK_PORT)
        self.execute_shell_command("./wizard_mock %d" % WIZARD_MOCK_PORT, "wizard_mock", wait=True, port=WIZARD_MOCK_PORT)

        if self.Parameters.processor_behaviour == PROCESSOR_MOCK:
            self.execute_shell_command(RUN_PROCESSOR_MOCK_CMD, "processor_mock", wait=True, port=ANTIROBOT_PROCESSOR_PORT)

        if self.Parameters.prod_antirobot:
            self.generate_config("antirobot", ANTIROBOT_CACHER_PORT, ANTIROBOT_ADMIN_PORT, ANTIROBOT_PROCESSOR_PORT, [ANTIROBOT_PROCESSOR_PORT, ANTIROBOT_PROCESSOR_PORT2])
            self.generate_config("antirobot_prod", ANTIROBOT_CACHER_PORT2, ANTIROBOT_ADMIN_PORT2, ANTIROBOT_PROCESSOR_PORT2, [ANTIROBOT_PROCESSOR_PORT, ANTIROBOT_PROCESSOR_PORT2])
            self.run_antirobot_daemon("antirobot_prod", ANTIROBOT_CACHER_PORT2)
        else:
            self.generate_config("antirobot", ANTIROBOT_CACHER_PORT, ANTIROBOT_ADMIN_PORT, ANTIROBOT_PROCESSOR_PORT, [ANTIROBOT_PROCESSOR_PORT])
        self.run_antirobot_daemon("antirobot", ANTIROBOT_CACHER_PORT)

        shutil.copytree("logs", "logs_early")
        sdk2.ResourceData(AntirobotTankLoadTestingLogsResource(self, "Antirobot stderr before tank", "logs_early")).ready()

        self.execute_shell_command(RUN_UNISTAT_PROXY_CMD, "unistat_proxy", wait=True, port=UNISTAT_PROXY_PORT)

        self.execute_shell_command("tank/bin/python tank/bin/yandex-tank ammo.txt -c load.ini",
                                   "tank", wait=True, callback=self.get_tank_ui_output)

        self.Context.testenv_values = self.get_testenv_values()

        sdk2.ResourceData(AntirobotTankLoadTestingLogsResource(self, "Antirobot stderr after tank", "logs")).ready()

        # Check services is OK
        assert check_port(ANTIROBOT_CACHER_PORT)
        if self.Parameters.prod_antirobot:
            assert check_port(ANTIROBOT_CACHER_PORT2)

        # To prevent errors like:
        # File local('current-antirobot_events.log') mtime or size changed during hashing!
        shutdown_antirobot(ANTIROBOT_ADMIN_PORT)
        if self.Parameters.prod_antirobot:
            shutdown_antirobot(ANTIROBOT_ADMIN_PORT2)

        if self.Parameters.make_flamegraph:
            self.make_flamegraph()

        sdk2.ResourceData(AntirobotTankLoadTestingLogsResource(self, "Antirobot logs artifacts", "antirobot/logs")).ready()
        sdk2.ResourceData(AntirobotTankLoadTestingLogsResource(self, "Antirobot data artifacts", "antirobot/data")).ready()
        if self.Parameters.prod_antirobot:
            sdk2.ResourceData(AntirobotTankLoadTestingLogsResource(self, "Antirobot-prod logs artifacts", "antirobot_prod/logs")).ready()
            sdk2.ResourceData(AntirobotTankLoadTestingLogsResource(self, "Antirobot-prod data artifacts", "antirobot_prod/data")).ready()

        self.Context.job_no = int(file_utils.read_file("logs/results/jobno_file.txt"))
        if self.Parameters.make_flamegraph:
            self.Context.flamegraph_url = "https://proxy.sandbox.yandex-team.ru/task/{}/logs/graph.svg".format(self.id)

    def make_flamegraph(self):
        arcadia = "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia"
        Arcadia.export("%s/contrib/tools/flame-graph/stackcollapse-perf.pl" % arcadia, "stackcollapse-perf.pl")
        Arcadia.export("%s/contrib/tools/flame-graph/flamegraph.pl" % arcadia, "flamegraph.pl")
        # perf script | ./stackcollapse-perf.pl | c++filt > out.perf-folded
        script_popen = sp.Popen(["perf", "script"], stdout=sp.PIPE)
        stackcollapse_popen = sp.Popen(["perl", "stackcollapse-perf.pl"], stdin=script_popen.stdout, stdout=sp.PIPE)
        with open("out.perf-folded", "w") as f:
            cppfilt_popen = sp.Popen(["c++filt"], stdin=stackcollapse_popen.stdout, stdout=f)
            cppfilt_popen.communicate()
        # ./flamegraph.pl out.perf-folded > logs/graph.svg
        with open("logs/graph.svg", "w") as f:
            sp.Popen(["perl", "flamegraph.pl", "out.perf-folded"], stdout=f).communicate()

        sdk2.ResourceData(AntirobotTankLoadTestingPerfDataResource(self, "perf data artifacts", "perf.data")).ready()

    def get_tank_ui_output(self):
        lines = get_shell_command_output("tail -n 1000 log1/common.log").split('\n')
        idx = len(lines) - 1
        while idx >= 0:
            line = lines[idx]
            if 'RPS:' in line and 'HTTP codes:' in line and 'Duration:' in line:
                break
            idx -= 1
        idx -= 2

        display_lines = []

        found = False

        while idx >= 0:
            line = lines[idx]
            tank_log_tag = 'INFO   (tank)'
            if tank_log_tag in line:
                cut_index = line.index(tank_log_tag) + len(tank_log_tag)
                display_lines.append(line[cut_index:])
            idx -= 1
            prev_line = display_lines[-2] if len(display_lines) > 1 else ""
            if 'RPS:' in prev_line and 'HTTP codes:' in prev_line and 'Duration:' in prev_line:
                found = True
                break

        if found:
            self.Context.tank_ui_output = "<pre>" + "\n".join(display_lines[::-1]).replace('<a', '&lt;a') + "</pre>"
            self.Context.tank_is_started = True
        else:
            if 'tank_is_started' not in self.Context or not self.Context.tank_is_started:
                self.Context.tank_ui_output = "<pre>Tank is not started yet</pre>"

        self.Context.save()

    @sdk2.header()
    def header(self):
        if self.Context.job_no:
            url = "https://lunapark.yandex-team.ru/" + str(self.Context.job_no)
            html = "<a href=\"" + url + "\">" + url + "</a><br/>"
            signals = json.loads(self.Context.testenv_values)
            if 'rps_real' in signals:
                html += "<b>Real RPS: " + str(signals['rps_real']) + "</b><br/>"
            html += "<pre>" + self.Context.tank_ui_output + "</pre><br/>"
            if self.Context.flamegraph_url:
                html += "<a href=\"" + self.Context.flamegraph_url + "\">Flame graph</a><br/>"
            return html
        else:
            if 'tank_ui_output' in self.Context:
                return self.Context.tank_ui_output
            else:
                return 'Initializing...'
