import json
import logging
import urllib

from sandbox import sdk2, common
import sandbox.common.types.task as ctt

from sandbox.projects.antirobot.MergeJsonConfigFiles import FindLastCorrectResourse
from sandbox.projects.antirobot.ValidationServiceJsonFile import AntirobotValidationServiceJsonFile


def get_last_correct_json(service_name):
    correct_service_config_resource = FindLastCorrectResourse(service_name)
    if correct_service_config_resource is not None:
        with open(str(sdk2.ResourceData(correct_service_config_resource).path), 'r') as jsonFile:
            return json.load(jsonFile)

    return None


def get_rules_from_url(url):
    lines = urllib.urlopen(url).read().split('\n')
    rules = []
    for line in lines:
        parts = line.split()
        rule = {}
        if len(parts) == 2:
            rule['path'] = parts[0]
            rule['req_type'] = parts[1]
            # CAPTCHA-1537, temporary crutch
            if rule['req_type'] == "main":
                rule['req_type'] = "analytical"
        elif len(parts) == 3:
            rule['path'] = parts[0]
            rule['cgi'] = parts[1]
            rule['req_type'] = parts[2]
            # CAPTCHA-1537, temporary crutch
            if rule['req_type'] == "main":
                rule['req_type'] = "analytical"
        else:
            assert len(parts) in (2, 3), "Should contain 2 or 3 fields. Line: " + line

        rules.append(rule)
    return rules


class AntirobotTransformRequestClassifierRulesToJsonServiceConfig(sdk2.Task):
    """Task to transform request classifier rules to json service config"""

    class Parameters(sdk2.Task.Parameters):
        service_name = sdk2.parameters.String('Service name', required=True)
        rules_urls = sdk2.parameters.String('Rules urls', required=True, multiline=True)
        rest_parameters = sdk2.parameters.JSON('Rest parameters', required=True)

    def create_json_config(self):
        urls = str(self.Parameters.rules_urls).split()
        json_config = self.Parameters.rest_parameters
        service = self.Parameters.service_name

        rules = []

        for url in urls:
            logging.debug('Retrieve rules for %s from url', service, url)
            rules.extend(get_rules_from_url(url))

        json_config['re_queries'] = rules
        return json_config

    def on_execute(self):
        with self.memoize_stage.create_children:
            service_name = self.Parameters.service_name
            json_config = self.create_json_config()

            last_correct_json = get_last_correct_json(service_name)

            if last_correct_json is None:
                logging.info("There is no correct json config for {}".format(service_name))

            if last_correct_json == json_config:
                logging.info("Last correct resource for {} has not been changed".format(service_name))
                return

            task = AntirobotValidationServiceJsonFile(self,
                                                      service_name=service_name,
                                                      json_config='json_strings',
                                                      json_string=json_config,
                                                      create_resource=True)
            task.enqueue()

            raise sdk2.WaitTask(task.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        failed_subtasks = [t for t in self.find().limit(0) if t.status in ctt.Status.Group.SCHEDULER_FAILURE]
        if not failed_subtasks:
            return

        raise common.errors.TaskError(
            '\n'.join(['Subtask {} ({}) was finished with the status of {}'.format(t.type, t.id, t.status) for t in
                       failed_subtasks])
        )
