import urllib
import json

from sandbox import sdk2

from sandbox.projects.antirobot.DiffResources import compare_multiple_resources
from sandbox.projects.antirobot.MergeJsonConfigFiles import RunJsonConfigChecker, UnitJsonConfigData, ANTIROBOT_TEAM
from sandbox.projects.common.nanny import nanny

URL_TO_JSON_PARAMETER = 'url_to_json'
JSON_STRINGS_PARAMETER = 'json_strings'
SUPERADMINS = list(ANTIROBOT_TEAM)
TEAM = SUPERADMINS


class AntirobotServiceIdentifierJsonFileResource(sdk2.Resource):
    """Service identifier json config file"""
    any_arch = True
    executable = False
    releasable = True
    releasers = TEAM


class AntirobotUpdateServiceIdentifier(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Task to validate and update service identifier json file"""

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Json config", required=True) as json_config:
            json_config.values[URL_TO_JSON_PARAMETER] = json_config.Value('Url to json config file', default=True)
            json_config.values[JSON_STRINGS_PARAMETER] = json_config.Value('Json config as string')

            with json_config.value[URL_TO_JSON_PARAMETER]:
                json_file_url = sdk2.parameters.String('Url to json config file', required=True)
            with json_config.value[JSON_STRINGS_PARAMETER]:
                json_string = sdk2.parameters.JSON('Json config as string', required=True)

        create_resource = sdk2.parameters.Bool('Create resource', required=True)
        dont_validate = sdk2.parameters.Bool('Do not validate (only for superadmins)', required=True)

    @staticmethod
    def check_json_file(json_file_name):
        json_data = UnitJsonConfigData()
        RunJsonConfigChecker(json_data)

    def check_access(self):
        author_name = self.author
        return author_name in TEAM

    def on_execute(self):
        json_file_name = 'service_identifier.json'
        if self.Parameters.json_config == URL_TO_JSON_PARAMETER:
            urllib.urlretrieve(self.Parameters.json_file_url, json_file_name)
        elif self.Parameters.json_config == JSON_STRINGS_PARAMETER:
            with open(json_file_name, 'w') as file:
                json.dump(self.Parameters.json_string, file, indent=4)

        if self.Parameters.dont_validate:
            assert self.author in SUPERADMINS, 'You are not a superadmin'
        else:
            self.check_json_file(json_file_name)

        if self.Parameters.create_resource:
            assert self.check_access(), 'Permission denied: you do not have rights for creating resource'

            resources_to_compare = list(sdk2.Resource.find(type=AntirobotServiceIdentifierJsonFileResource, state='READY').limit(10))

            description = "Json service identifier config for antirobot"
            resource = AntirobotServiceIdentifierJsonFileResource(self, description, json_file_name, ttl='inf')
            sdk2.ResourceData(resource).ready()

            self.Context.diffs = compare_multiple_resources(self, resources_to_compare, resource, serializable=True)

    @sdk2.header()
    def header(self):
        if 'diffs' not in self.Context or self.Context.diffs is None or len(self.Context.diffs) == 0:
            return None

        html = ''
        for diff in self.Context.diffs:
            diff_resource_url = diff[2]['http_proxy']
            old_resource_id = diff[0]['id']
            new_resource_id = diff[1]['id']
            html += '<a href="{}">Compare {} and {}</a>'.format(diff_resource_url, old_resource_id, new_resource_id)
            html += '<br/>'
            html += '\n'
        return html
