import urllib
import json

from sandbox import sdk2

from sandbox.projects.antirobot.MergeJsonConfigFiles import SERVICE_LOGINS, ANTIROBOT_TEAM, RunJsonConfigChecker, UnitJsonConfigData

URL_TO_JSON_PARAMETER = 'svn_url_to_json'
JSON_STRINGS_PARAMETER = 'json_strings'
SUPERADMINS = list(ANTIROBOT_TEAM)


class AntirobotServiceJsonFileResource(sdk2.Resource):
    """Service json config file"""
    any_arch = True
    executable = False
    releasable = True
    releasers = None
    service = sdk2.Attributes.String(required=True)


class AntirobotValidationServiceJsonFile(sdk2.Task):
    """Task to validate service's json file"""

    class Parameters(sdk2.Task.Parameters):
        service_name = sdk2.parameters.String('Service name', required=True)

        with sdk2.parameters.String("Json config", required=True) as json_config:
            json_config.values[URL_TO_JSON_PARAMETER] = json_config.Value('Url to json config file', default=True)
            json_config.values[JSON_STRINGS_PARAMETER] = json_config.Value('Json config as string')

            with json_config.value[URL_TO_JSON_PARAMETER]:
                json_file_url = sdk2.parameters.String('Url to json config file', required=True)
            with json_config.value[JSON_STRINGS_PARAMETER]:
                json_string = sdk2.parameters.JSON('Json config as string', required=True)

        create_resource = sdk2.parameters.Bool('Create resource', required=True)
        dont_validate = sdk2.parameters.Bool('Do not validate (only for superadmins)', required=True)

    @staticmethod
    def check_json_file(json_file_name, service):
        with open(json_file_name, 'r') as json_file:
            json_data = json.load(json_file)
            assert isinstance(json_data, dict), "Json file should contain map"
            assert 'service' in json_data, "Json file should contain field 'service'"
            assert json_data['service'] == service, "Incorrect service name. " \
                                                    "Expected {}, but got {}".format(service, json_data['service'])

        json_data = UnitJsonConfigData()
        with open(json_file_name, 'r') as json_file:
            for i in range(len(json_data)):
                if json_data[i]['service'] == service:
                    json_data[i] = json.load(json_file)
        RunJsonConfigChecker(json_data)

    def check_access(self):
        author_name = self.author
        return author_name in SERVICE_LOGINS[self.Parameters.service_name]

    def on_execute(self):
        service_name = self.Parameters.service_name
        json_file_name = service_name + '_service_config.json'
        if self.Parameters.json_config == URL_TO_JSON_PARAMETER:
            urllib.urlretrieve(self.Parameters.json_file_url, json_file_name)
        elif self.Parameters.json_config == JSON_STRINGS_PARAMETER:
            with open(json_file_name, 'w') as file:
                json.dump(self.Parameters.json_string, file, indent=4)

        if self.Parameters.dont_validate:
            assert self.author in SUPERADMINS, 'You are not a superadmin'
        else:
            self.check_json_file(json_file_name, service_name)

        if self.Parameters.create_resource:
            assert self.check_access(), 'Permission denied: you do not have rights for creating resource'

            description = "Json config file for service %s" % service_name
            resource = AntirobotServiceJsonFileResource(self, description, json_file_name, service=service_name, ttl='inf')
            sdk2.ResourceData(resource).ready()
