import logging
from sandbox import sdk2
from sandbox.common.types import task as ctt
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.gencfg.AllocateInDynamic import AllocateInDynamic
from itertools import product


class AllocateAppHostYappySlotsInDynamic(sdk2.Task):
    """Allocate Yappy betas for app_host,src_setup,http_adapter"""

    vertical_to_prj = {
        "WEB": "web-main",
        "VIDEO": "video-main",
        "IMGS": "imgs-main",
        "ATOM": 'web-mobreport-atomsearch',
        "COMMON": "apphostcommon",
        "MAIL": "mail",
        "MAILCORP": "mailcorp",
        "SHARED": "shared",
        "NEWS": "news"
    }

    volumes = [
        {
            "symlinks": [],
            "guest_mp": "",
            "host_mp_root": "/place",
            "quota": "15 Gb"
        },
        {
            "symlinks": [],
            "guest_mp": "/",
            "host_mp_root": "/place",
            "quota": "5 Gb"
        },
        {
            "symlinks": ["/usr/local/www/logs"],
            "guest_mp": "/logs",
            "host_mp_root": "/place",
            "quota": "24 Gb"
        }
    ]

    owners = ['librarian', 'sharpeye', 'pmatsula']

    ctype = "priemka-apphost"

    geos = ['sas', 'man', 'vla']

    app_to_itype = {
        "app_host": "apphost",
        "src_setup": "ahadapter",
        "http_adapter": "httpadapter"
    }

    app_to_gencfg_name = {
        "app_host": "APPHOST",
        "src_setup": "SRC_SETUP",
        "http_adapter": "HTTP_ADAPTER"
    }

    app_to_requriments = {
        "app_host": {
            "instance_count": 2,
            "cores_per_instance": 1,
            "memory": 4,
            "disk": 100,
            "ssd": 0,
            "net": 0,
            "volumes": volumes
        },
        "src_setup": {
            "instance_count": 2,
            "cores_per_instance": 1,
            "memory": 4,
            "disk": 100,
            "ssd": 0,
            "net": 0,
            "volumes": volumes
        },
        "http_adapter": {
            "instance_count": 2,
            "cores_per_instance": 1,
            "memory": 4,
            "disk": 100,
            "ssd": 0,
            "net": 0,
            "volumes": volumes
        }
    }

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1000

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.CheckGroup("Choose vertical", required=True) as choose_vertical:
            choose_vertical.values.WEB = choose_vertical.Value("WEB", checked=False)
            choose_vertical.values.VIDEO = choose_vertical.Value("VIDEO", checked=False)
            choose_vertical.values.IMGS = choose_vertical.Value("IMGS", checked=False)
            choose_vertical.values.ATOM = choose_vertical.Value("ATOM", checked=False)
            choose_vertical.values.COMMON = choose_vertical.Value("COMMON", checked=False)
            choose_vertical.values.MAIL = choose_vertical.Value("MAIL", checked=False)
            choose_vertical.values.MAILCORP = choose_vertical.Value("MAILCORP", checked=False)
            choose_vertical.values.SHARED = choose_vertical.Value("SHARED", checked=False)
            choose_vertical.values.NEWS = choose_vertical.Value("NEWS", checked=False)

        with sdk2.parameters.CheckGroup("Choose app", required=True) as choose_app:
            choose_app.values.app_host = choose_vertical.Value("app_host", checked=False)
            choose_app.values.http_adapter = choose_vertical.Value("src_setup", checked=False)
            choose_app.values.src_setup = choose_vertical.Value("http_adapter", checked=False)

        beta_idx = sdk2.parameters.String("Beta indices (list: 1 2 5 8)", required=True)
        ticket = sdk2.parameters.String("Ticket", required=True)
        dry_run = sdk2.parameters.Bool("Dry run", required=True, default=True)

    def on_execute(self):
        with self.memoize_stage.create_tasks:
            logging.info("Create tasks")

            params = product(
                self.Parameters.choose_vertical,
                self.Parameters.choose_app,
                self.Parameters.beta_idx.split(" "),
                self.geos
            )
            for vertical, app, idx, geo in params:

                gencfg_group = "{vertical}_{gencfg_app_name}_YAPPY_TESTING_{idx}".format(
                    vertical=vertical,
                    gencfg_app_name=self.app_to_gencfg_name.get(app),
                    idx=idx
                ).upper()

                full_gencfg_group = "{}_{}".format(geo, gencfg_group).upper()
                service_name = "yappy_{app}_{geo}_{vertical}_testing_{idx}".format(
                    vertical=vertical,
                    app=app,
                    idx=idx,
                    geo=geo
                ).lower()

                additional_prj = service_name.replace("_", "-")
                app_reqs = self.app_to_requriments.get(app)

                subtask = AllocateInDynamic(
                    self,
                    description=u'Generating groups for Yappy betas {}'.format(self.Parameters.ticket)
                )

                # Fill params
                subtask.Parameters.author = self.author
                subtask.Parameters.group_name = gencfg_group
                subtask.Parameters.group_descr = 'Yappy groups for {} betas {}'.format(app, self.Parameters.ticket)
                subtask.Parameters.owners = self.owners
                subtask.Parameters.watchers = []

                subtask.Parameters.ctype = self.ctype
                subtask.Parameters.itype = self.app_to_itype.get(app)
                subtask.Parameters.prj = [self.vertical_to_prj.get(vertical), additional_prj]
                subtask.Parameters.metaprj = 'unknown'
                subtask.Parameters.itags = ''
                subtask.Parameters.affinity_category = None

                # Requirements
                subtask.Parameters.instance_count = app_reqs.get("instance_count")
                subtask.Parameters.cores_per_instance = app_reqs.get("cores_per_instance")
                subtask.Parameters.memory = app_reqs.get("memory")
                subtask.Parameters.disk = app_reqs.get("disk")
                subtask.Parameters.ssd = app_reqs.get("ssd")
                subtask.Parameters.net = app_reqs.get("net")

                # Locations
                subtask.Parameters.locations = [geo]

                subtask.Parameters.volumes = app_reqs.get("volumes")
                subtask.Parameters.parent_macro = ''

                subtask.Parameters.commit_message = 'Allocate {} {}'.format(full_gencfg_group, self.Parameters.ticket)
                subtask.Parameters.dry_run = self.Parameters.dry_run
                subtask.Parameters.arcadia_revision = ''
                subtask.Parameters.use_last_resources = False

                subtask.save()
                task = subtask.enqueue()

                logging.debug(self.Context.allocate_task_ids)
                if not self.Context.allocate_task_ids:
                    self.Context.allocate_task_ids = [task.id]
                else:
                    self.Context.allocate_task_ids.append(task.id)
                logging.debug(self.Context.allocate_task_ids)

            logging.debug("allocated task ids {}".format(self.Context.allocate_task_ids))

            raise sdk2.WaitTask(
                self.Context.allocate_task_ids,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        with self.memoize_stage.check_results:
            sub_tasks = self.find(status=ctt.Status.Group.BREAK + ctt.Status.Group.FINISH).limit(100)
            for task in sub_tasks:
                if task.status not in [ctt.Status.SUCCESS]:
                    raise SandboxTaskFailureError("Task {} failed".format(task.id))
