import datetime as dt
import json
import logging
import os
import requests

from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import NotExists
from sandbox.common.types.resource import State
import sandbox.common.types.task as ctt
from sandbox.projects.common.betas.yappy_api import YappyApi
from sandbox.projects.common.nanny.client import NannyClient
from sandbox.projects.tank.ShootViaTankapi import ShootViaTankapi
from sandbox.sandboxsdk.svn import Arcadia, Svn
from sandbox import sdk2
import sandbox.projects.release_machine.core.const as rm_const
import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.helpers.startrek_helper as st_helper
import sandbox.projects.release_machine.components.all as rmc
from sandbox.projects.tank.load_resources.resources import AMMO_FILE

NANNY_URL = 'http://nanny.yandex-team.ru'
YAPPY_API_URL = 'https://yappy.z.yandex-team.ru/'
DEFAULT_TANK = 'nanny:production_yandex_tank'


class AppHostLoadTest(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        tank_names = sdk2.parameters.List(
            "Tanks",
            required=True,
            default=DEFAULT_TANK,
        )
        beta = sdk2.parameters.String(
            "Yappy beta",
            required=False
        )
        type_or_quota = sdk2.parameters.String(
            "Component type or quota name",
            required=False
        )
        service = sdk2.parameters.String(
            "Nanny service to test",
            required=False,
            description='Required if the previous two parameters are not set'
        )
        load_plan = sdk2.parameters.String(
            "Tank load plan",
            required=True,
            description='e.g. line(1,10,30s) const(10,10m) step(10, 20, 5, 10s)'
        )
        queries = sdk2.parameters.Resource(
            "Queries for service in ammo format",
            required=False,
        )
        tests_cgi = sdk2.parameters.List(
            "List of cgi queries for shooting"
        )
        rps_to_survive = sdk2.parameters.Integer(
            "Minimal RPS to survive",
            default=0,
        )
        time_threshold = sdk2.parameters.Integer(
            "Max value for 0.95 response quantile, ms",
            default=50,
            description="If 0.95-quantile value is greater for X rps, task considers that service cannot survive X rps",
        )
        report_to_release_ticket = sdk2.parameters.Bool(
            "Report to release ticket",
            default=False
        )
        with report_to_release_ticket.value[True]:
            rm_component = sdk2.parameters.String(
                "Release machine component"
            )
            default_release_ticket = sdk2.parameters.String(
                "Default ticket for report",
                required=False,
            )
        ticket = sdk2.parameters.String(
            "Startrek ticket connected to this shooting",
            required=True
        )
        nanny_token = sdk2.parameters.String(
            "Nanny token vault name for the owner",
            required=True,
            default='nanny_oauth_token'
        )

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Context(sdk2.Context):
        ammo = {}
        results_by_label = {}
        ammo_files_created = 0
        restart_pool = 3
        did_full_retry = False

    def get_next_shooting(self):
        for label in self.Context.ammo:
            if self.Context.results_by_label.get(label, None) is None:
                return label

        return None

    def find_in_sandbox(self, label):
        try:
            res = sdk2.Resource.find(
                resource_type=AMMO_FILE,
                attrs={'ammo_label': label},
                status=State.READY,
            ).first()
            if res.state in [State.BROKEN, State.DELETED]:
                return None
            self.Context.ammo = res.id
            logging.debug("Found ammo file with id {}, no need to convert input resource".format(res.id))
            return res.id
        except:
            return None

    def get_nanny_service(self):
        if self.Parameters.service:
            return self.Parameters.service

        beta = self.Parameters.beta
        type_or_quota = self.Parameters.type_or_quota
        if not beta or not type_or_quota:
            raise TaskFailure("Set either nanny service or (beta, type_or_quota)")

        api = YappyApi(YAPPY_API_URL, ssl_verify=False)
        if not api.beta_exists(beta):
            raise TaskFailure("Beta with name {} does not exist.".format(beta))

        beta_info = api._make_req_post(
            '/api/yappy.services.Model/retrieveBeta',
            data=json.dumps({'name': beta, 'withHistory': False})
        )

        logging.debug("Trying to find  component with quota or type name: {}".format(self.Parameters.type_or_quota))
        for comp in beta_info['components']:
            quota = comp.get('slot', {}).get('quotaName')
            comp_type = comp.get('type', {}).get('name')
            logging.debug("Found component with type {} and quota {}".format(comp_type, quota))
            if quota == type_or_quota or comp_type == type_or_quota:
                service = comp.get('slot', {}).get('id')
                logging.debug("Task will shoot to service {}".format(service))
                return service

        raise TaskFailure("Task failed to find a component with required type or quota name")

    def build_cgi_queries_resource(self, cgi):
        self.Context.ammo_files_created += 1
        output_path = 'queries_{}.ammo'.format(self.Context.ammo_files_created)
        with open(output_path, "w") as fout:
            fout.write(cgi)

        output = AMMO_FILE(
            self,
            "Cgi queries for cgi {}".format(cgi),
            output_path,
        )
        sdk2.ResourceData(output).ready()
        return output.id

    def build_shooting_config(self, nanny_service, label):
        arcadia_path = Arcadia.checkout('svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia', 'arcadia', depth=Svn.Depth.IMMEDIATES)
        config_template_path = os.path.join(arcadia_path, 'sandbox/projects/app_host/AppHostLoadTest/shooting_config.txt')
        Arcadia.update(config_template_path, depth=Svn.Depth.IMMEDIATES, parents=True)
        with open(config_template_path, 'r') as f:
            nanny = NannyClient(NANNY_URL, self.Parameters.nanny_token)
            instance_info = nanny.get_service_current_instances(nanny_service)['result'][0]
            config = f.read()
            config = config.replace("<ADDRESS>", "{}:{}".format(instance_info['container_hostname'], instance_info['port']))
            config = config.replace("<AMMOFILE>", "https://proxy.sandbox.yandex-team.ru/{}".format(self.Context.ammo[label]))
            config = config.replace("<LOAD_PLAN>", self.Parameters.load_plan)
            config = config.replace("<DESCRIPTION>", self.Parameters.description)
            config = config.replace("<AUTHOR>", self.author)
            config = config.replace("<TICKET>", self.Parameters.ticket)
            logging.debug("Tank config is:\n{}".format(config))
            return config, instance_info['port'] + 4

    def get_request(self, api):
        answer = requests.get(api, timeout=10)
        try:
            return json.loads(answer.text)
        except:
            return {}

    def report_to_ticket(self):
        TARGET_RPS_FAST_ERROR = 100
        TARGET_RPS_LONG_ANSWER = 150
        TARGET_RPS_SHORT_ANSWER = 250

        OK = "**!!(green)OK!!**"
        CRITICAL = "**!!(red)CRITICAL!!**"

        branch = None
        tags = self.server.task[self.id].read()["tags"]
        for tag in tags:
            try:
                branch = int(tag.split('-')[-1])
            except:
                pass
        if branch is None:
            logging.debug("Task failed to find its branch tag. Tags: {}".format(tags))
            return

        results = []
        for tag in self.Context.shooting_results:
            rps = self.Context.shooting_results[tag]
            lunapark_link = '(({} link))'.format(self.Context.lunapark_links[tag])
            target = 0
            if 'fast_error=1' in tag:
                tag = 'Fast errors'
                target = TARGET_RPS_FAST_ERROR
            elif 'length' in tag:
                length = tag.split('length=')[-1].split('&')[0]
                tag = 'Responses of length {}'.format(length)
                target = TARGET_RPS_LONG_ANSWER if int(length) > 1000 else TARGET_RPS_SHORT_ANSWER

            result = OK if rps >= target else CRITICAL
            results.append({
                'tag': tag,
                'rps': rps,
                'link': lunapark_link,
                'target': target,
                'result': result,
            })

        logging.debug('Results list for startrek: {}'.format(results))

        message = '\n'.join([
            '#|',
            '|||**Shooting description**|**Survived RPS**|**Target RPS**|**Lunapark link**||'] + [
            '||{result}|{tag}|{rps}|{target}|{link}||'.format(**r) for r in results
            ] + [
            '|#'
        ])
        logging.debug("Message for startrek: {}".format(message))

        st = st_helper.STHelper(sdk2.Vault.data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
        c_info = rmc.COMPONENTS[self.Parameters.rm_component]()
        group_name = "====Tank perf test"
        title = ""
        if self.Parameters.default_release_ticket:
            st.write_grouped_comment_in_issue(
                self.Parameters.default_release_ticket,
                group_name,
                title,
                message,
                None,
                False,
            )
        else:
            st.write_grouped_comment(
                group_name,
                title,
                message,
                branch,
                c_info,
            )

    def get_max_rps(self, job_id, start_ts, finish_ts):
        def dt_from_str(s):
            return dt.datetime.strptime(s, '%Y-%m-%d %H:%M:%S')

        def int_from_dt(d):
            return int(d.strftime("%Y%m%d%H%M%S"))

        start_ts = dt_from_str(start_ts)
        finish_ts = dt_from_str(finish_ts)
        rps_stat_api = "https://lunapark.yandex-team.ru/api/job/{}/data/req_cases.json".format(job_id)
        rps_stats = self.get_request(rps_stat_api)
        rps_by_time = {int_from_dt(dt_from_str(item['dt'])): item['cnt'] for item in rps_stats}
        quant_by_rps = {}

        cur_dt = start_ts
        while cur_dt <= finish_ts:
            ts = int_from_dt(cur_dt)
            perc_api = "https://lunapark.yandex-team.ru/api/job/{j}/dist/percentiles.json?percents=95&from={ts}&to={ts}".format(j=job_id, ts=ts)
            http_api = "https://lunapark.yandex-team.ru/api/job/{j}/dist/http.json?from={ts}&to={ts}".format(j=job_id, ts=ts)
            perc_json = self.get_request(perc_api)
            http_json = self.get_request(http_api)
            if not perc_json or not http_json:
                self.set_info("WARNING: task failed to find stats for timestamp {}".format(ts))
            else:
                quant_by_rps[rps_by_time[ts]] = perc_json[0]['ms']
            cur_dt += dt.timedelta(seconds=1)

        last_good = 1
        for rps in quant_by_rps:
            if quant_by_rps[rps] <= self.Parameters.time_threshold and last_good < rps:
                last_good = rps

        return last_good

    def check_subtasks(self):
        if self.Context.shooting_task_ids is NotExists:
            return None
        for id in self.Context.shooting_task_ids:
            task = sdk2.Task.find(id=id).first()
            if task.status == ctt.Status.EXCEPTION or task.status == ctt.Status.TIMEOUT:
                return task
        return None

    def restart_broken_subtask(self, task):
        if self.Context.restart_pool <= 0:
            if not self.Context.did_full_retry:
                self.do_full_retry()
            else:
                raise TaskFailure("Child task {} failed. No restarts left".format(task.id))
        self.set_info("Shooting #{} failed. Trying to restart".format(task.id))
        self.Context.restart_pool -= 1
        task.enqueue()
        raise sdk2.WaitTask(task.id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def on_execute(self):
        if not len(self.Context.ammo):
            if self.Parameters.queries:
                self.Context.ammo['attached_queries'] = self.Parameters.queries.id
            for cgi in self.Parameters.tests_cgi:
                self.Context.ammo[cgi] = self.build_cgi_queries_resource(cgi)

        broken_task = self.check_subtasks()
        if broken_task is not None:
            self.restart_broken_subtask(broken_task)

        label = self.get_next_shooting()
        if label is not None:
            nanny_service = self.get_nanny_service()
            config, port = self.build_shooting_config(nanny_service, label)
            if DEFAULT_TANK in self.Parameters.tank_names:
                shooting_task_id = ShootViaTankapi(
                    self,
                    nanny_service=nanny_service,
                    override_nanny_port=port,
                    use_public_tanks=True,
                    tanks=self.Parameters.tank_names,
                    config_source='file',
                    config_content=config,
                    kill_timeout=10800,
                ).enqueue().id
            else:
                shooting_task_id = ShootViaTankapi(
                    self,
                    use_public_tanks=False,
                    tanks=self.Parameters.tank_names,
                    config_source='file',
                    config_content=config,
                    kill_timeout=10800,
                ).enqueue().id
            if self.Context.shooting_task_ids is NotExists:
                self.Context.shooting_task_ids = [shooting_task_id]
            else:
                self.Context.shooting_task_ids.append(shooting_task_id)

            self.Context.results_by_label[label] = shooting_task_id
            raise sdk2.WaitTask(shooting_task_id, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)
        else:
            self.Context.lunapark_links = {}
            self.Context.shooting_results = {}
            for label in self.Context.results_by_label:
                task_id = self.Context.results_by_label[label]
                task = sdk2.Task.find(id=task_id, children=True).first()
                if task.status != ctt.Status.SUCCESS:
                    if not self.Context.did_full_retry:
                        self.do_full_retry()
                    else:
                        raise Exception("Shooting task {} failed".format(task_id))
                job_id = task.Parameters.lunapark_job_id
                lunapark_link = task.Parameters.lunapark_link
                self.Context.lunapark_links[label] = lunapark_link

                api = "https://lunapark.yandex-team.ru/api/job/{}/aggregates.json".format(job_id)
                stat = self.get_request(api)[0]
                time_from = stat['trail_start']
                time_to = stat['trail_stop']
                report = [
                    "Shooting for label: {}".format(label),
                    "Start time: {}".format(time_from),
                    "End time:   {}".format(time_to),
                ]
                for val in [50, 90, 95, 99]:
                    report.append("Quantile {}: {} ms".format(val, stat['q{}'.format(val)]))
                report.append("HTTP 200: {}%".format(stat.get('http_200_percent', 0)))

                max_rps = self.get_max_rps(job_id, time_from, time_to)
                self.Context.shooting_results[label] = max_rps

                report.append("Estimated max rps for service: {}".format(max_rps))
                report.append('Lunapark link: <a href="{link}">{link}</a>'.format(link=lunapark_link))
                self.set_info('<br>'.join(report), do_escape=False)

            try:
                if self.Parameters.report_to_release_ticket:
                    self.report_to_ticket()
            except Exception as e:
                self.set_info("WARNING: task failed to report to startrek ticket. Exception text: {}".format(e))

            for tag in self.Context.shooting_results:
                if self.Context.shooting_results[tag] < self.Parameters.rps_to_survive:
                    raise TaskFailure("Service failed to survive {} RPS".format(self.Parameters.rps_to_survive))
