# -*- coding: utf-8

from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2 import Task, Vault, ResourceData
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.service_resources import SandboxTasksBinary
from sandbox.sandboxsdk.environments import ArcEnvironment
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants
from sandbox.projects.resource_types import OTHER_RESOURCE as OtherResource
import sandbox.common.types.task as ctt
from tempfile import mkdtemp
from shutil import rmtree, move
import os
import datetime
import time
import json
import logging


class AppHostUpdateTests(Task):
    _built_binaries = {}

    def on_create(self):
        if self.Requirements.tasks_resource is None:
            self.Requirements.tasks_resource = SandboxTasksBinary.find(
                attrs={
                    'target': 'sandbox/projects/app_host/AppHostUpdateTests',
                    'released': ctt.ReleaseStatus.STABLE
                }
            ).first()

    def on_execute(self):
        with arcadiasdk.mount_arc_path('arcadia-arc:/#trunk') as arcadia:
            dumped_sessions, date = self._dump_sessions(arcadia)
            tests_path = self._generate_tests(arcadia, dumped_sessions)

            results = self._run_tests(arcadia, tests_path, False)

            bad_tests = self._get_bad_tests(results, os.path.relpath(tests_path, arcadia))

            if len(bad_tests) > 0:
                logging.info('Bad tests: {}'.format(bad_tests))
                resource = self._separate_bad_tests(dumped_sessions, bad_tests)
                self._dump_to_ticket(resource, date)

                if len(bad_tests) > 10:
                    raise Exception('too many broken tests')

                tests_path = self._generate_tests(arcadia, dumped_sessions)
                self._run_tests(arcadia, tests_path, True)

            self._upload_test_data(arcadia, tests_path)

            new_tests_path = os.path.join(os.path.dirname(tests_path), 'generated')
            if os.path.isdir(new_tests_path):
                rmtree(new_tests_path)
            os.rename(tests_path, new_tests_path)

            self._commit(new_tests_path)

    def _dump_to_ticket(self, resource, date):
        from startrek_client import Startrek
        import requests

        DESCRIPTION = '''При генерации новых тестов на основе логов обнаружены сессии, которые не удалось воспроизвести в тестах.
Sandbox таска: https://sandbox.yandex-team.ru/task/{}
Плохие сессии: https://sandbox.yandex-team.ru/resource/{}
Чтобы было проще найти ошибку, нужно изучить документацию: https://wiki.yandex-team.ru/apphost/Generiruemye-testy'''
        queue = 'APPHOST'
        component_id = 78215  # generated_tests
        startrek_token = Vault.data('APP_HOST', 'robot-ah-releases-startrek-token')
        abc_token = Vault.data('APP_HOST', 'robot-ah-releases-abc-token')
        sb_task = self.id
        resource = resource.id
        service_id = 1515

        client = Startrek(useragent='startrek_python_client', token=startrek_token)
        issues = client.issues.find('Queue: {} AND Components: {} AND Resolution: empty()'.format(queue, component_id))
        text = DESCRIPTION.format(sb_task, resource, '')
        if len(issues) > 0:
            issue = issues[0]
            issue.comments.create(text=text)
        else:
            shift = requests.get('https://abc-back.yandex-team.ru/api/v4/services/{}/on_duty/'.format(service_id),
                                 headers={'Authorization': 'OAuth {}'.format(abc_token)}).json()
            person = shift[0]['person']['login']
            issue = client.issues.create(queue=queue,
                                         summary='Генерируемые тесты {}'.format(date),
                                         description=text,
                                         components=[component_id],
                                         assignee=person)
        logging.info('Problems dumped to ticket {}'.format(issue.key))
        pass

    def _separate_bad_tests(self, dumped_sessions, bad_tests):
        bad_tests_path = self.path('bad_tests').as_posix()
        for test in bad_tests:
            parts = test.split('/')
            vertical = parts[0]
            vertical_path = os.path.join(bad_tests_path, vertical)
            if not os.path.exists(vertical_path):
                os.makedirs(vertical_path)
            move(os.path.join(dumped_sessions, test), vertical_path)

        resource = OtherResource(self, 'bad tests data', bad_tests_path)
        data = ResourceData(resource)
        data.ready()
        return resource

    def _get_bad_tests(self, results, tests_path):
        bad_tests = []
        for result in results:
            if result['type'] != 'test':
                continue

            if tests_path not in result['path']:
                continue

            if result['name'] != 'test.py':
                continue

            if result['status'] != 'OK':
                bad_tests.append(os.path.relpath(result['path'], tests_path))

        return bad_tests

    def _commit(self, new_tests_path):
        with ProcessLog(self, logger="arc") as arc_log:
            arc = ArcEnvironment().prepare()
            arc_token = Vault.data('APP_HOST', 'ARC_TOKEN')
            env = {
                'ARC_TOKEN': arc_token,
                'USER': 'robot-ah-releases'
            }
            now = datetime.datetime.now()
            branch_name = 'generated_tests_{}_{}_{}_{}'.format(now.year, now.month, now.day, int(time.time()))
            subprocess.Popen('{} co -b {}'.format(arc, branch_name), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=new_tests_path, env=env).wait()
            subprocess.Popen('{} add {}'.format(arc, new_tests_path), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=new_tests_path, env=env).wait()
            commit_msg = 'test(app_host): update generated tests'
            subprocess.Popen('{} ci -m "{}"'.format(arc, commit_msg), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=new_tests_path, env=env).wait()
            subprocess.Popen('{} pr create --push -A -m "{}"'.format(arc, commit_msg), shell=True, stdout=arc_log.stdout, stderr=arc_log.stderr, cwd=new_tests_path, env=env).wait()

    def _run_tests(self, arcadia, tests_path, check_rc):
        logging.info('Running tests')
        build_path = mkdtemp()
        arcadiasdk.do_build(
            build_system=sdk_constants.YA_MAKE_FORCE_BUILD_SYSTEM,
            source_root=arcadia,
            targets=[tests_path],
            test=True,
            clear_build=False,
            results_dir=build_path,
            check_rc=check_rc
        )

        results_path = build_path + os.sep + 'results.json'

        with open(results_path, 'r') as outfile:
            results = json.load(outfile)
            return results['results']

    def _dump_sessions(self, arcadia):
        logging.info('Fetching sessions')
        dump_fetcher_bin = self._build_binary(arcadia, 'apphost/tools/mock_framework/dump_fetcher')
        dumped_sessions = arcadia + '/apphost/test/legacy/dumped_sessions'
        dump_fetcher_params = [dump_fetcher_bin, '1h', 'last', '--count', '10', '-o', dumped_sessions]
        with ProcessLog(self, logger="dump_fetcher") as log:
            env = {
                'YT_TOKEN': Vault.data('APP_HOST', 'robot-ah-releases-yt-token')
            }
            if subprocess.Popen(' '.join(dump_fetcher_params), shell=True, stdout=log.stdout, stderr=log.stderr, env=env).wait():
                raise Exception('dump sessions failure')
            log.stdout.flush()
            with log.stdout.path.open() as log_file:
                line = log_file.readline()
                date = line.split("Date: ", 1)[1]

        return dumped_sessions, date

    def _generate_tests(self, arcadia, dumped_sessions):
        logging.info('Generating tests')
        generator_bin = self._build_binary(arcadia, 'apphost/tools/mock_framework/tests/generator')
        generator_params = [generator_bin, dumped_sessions]
        with ProcessLog(self, logger="tests_generator") as log:
            env = {
                'ARCANUM_TOKEN': Vault.data('APP_HOST', 'robot-ah-releases-arcanum-token')
            }
            if subprocess.Popen(' '.join(generator_params), shell=True, stdout=log.stdout, stderr=log.stderr, env=env).wait():
                raise Exception('generate test failure')
        return dumped_sessions + '_tests'

    def _upload_test_data(self, arcadia, tests_path):
        logging.info('Uploading tests')
        uploader_bin = self._build_binary(arcadia, 'apphost/tools/mock_framework/tests/generator/uploader')
        uploader_params = [uploader_bin, tests_path, 'inf']
        with ProcessLog(self, logger="tests_uploader") as log:
            env = {
                'SANDBOX_TOKEN': Vault.data('APP_HOST', 'robot-ah-releases-sandbox-token')
            }
            if subprocess.Popen(' '.join(uploader_params), shell=True, stdout=log.stdout, stderr=log.stderr, env=env).wait():
                raise Exception('upload test data failure')

    def _build_binary(self, arcadia, target):
        if target in self._built_binaries:
            return self._built_binaries[target]
        binary = os.path.basename(target)
        build_path = mkdtemp()
        arcadiasdk.do_build(
            build_system=sdk_constants.YA_MAKE_FORCE_BUILD_SYSTEM,
            source_root=arcadia,
            targets=[target],
            results_dir=build_path,
            clear_build=False,
        )

        buit_binary = build_path + os.sep + 'bin' + os.sep + binary
        self._built_binaries[target] = buit_binary
        return buit_binary
