import datetime
import logging

from sandbox.projects.app_host.BuildAppHostConfigBundle import BuildAppHostConfigBundle, BuildServiceResources, \
    BuildOnlyServiceResources, EnrichGraphsWithRevisions, BuildTemplatesDir, RunSmokeTest, NoBackEndsInGraph, AllowGraphRemoval
from sandbox.projects.common.app_host.options import get_verticals, ChooseVertical, GraphGenerator
from sandbox.projects.common.app_host.regular_build import save_run_info, has_changes_since_prev_run
from sandbox.projects.DeployNannyDashboard import ReleaseTask, NannyDashboardName, NannyDashboardRecipeName
from sandbox.projects.DeployNannyDashboard import NannyDashboardFilter, SandboxReleaseType, NannyWaitDeployParameter
from sandbox.projects.DeployNannyDashboard import VaultOwner, VaultName
from sandbox.projects.sandbox.who_is_on_duty_today.utils import CalendarAPI

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, SandboxSelectParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


DATE_FORMAT = "%Y-%m-%d"


class OptionalSandboxArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    required = False
    default_value = ''


class DeployVariant(SandboxSelectParameter):
    name = 'deploy_variant'
    description = "Choose which variant to deploy"
    required = True
    default_value = 'dev'
    choices = [
        ('dev', 'dev'),
        ('prod', 'prod')
    ]


def get_dashboard_name(vertical):
    if vertical == "WEB":
        return "app_host"
    else:
        return "app_host_{}".format(vertical.lower())


class AutoBuildAppHostConfigBundle(SandboxTask):
    type = "BUILD_APP_HOST_CONFIG_BUNDLE_AUTO"

    input_parameters = [
        ChooseVertical,
        GraphGenerator,
        DeployVariant,
        OptionalSandboxArcadiaUrlParameter,
        NoBackEndsInGraph
    ]

    IS_WORK_DAY = "is_work_day"
    DESCRIPTION_SEPARATOR = " | "

    deploy_settings = {
        "ATOM": {
            "prod": {
                "release": True,
                "testing_release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "enrich": True,
                "run_smoke_test": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": True,
                "deploy_recipe": "app_host_ppsa_auto"
            }
        },
        "WEB": {
            "prod": {
                "release": True,
                "testing_release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "enrich": True,
                "run_smoke_test": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto_with_locks"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": True,
                "deploy_recipe": "app_host_ppsa"
            }
        },
        "IMGS": {
            "prod": {
                "release": True,
                "testing_release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "enrich": True,
                "run_smoke_test": False,
                "delete_graphs": True,
                "vault_owner": "APP_HOST_IMGS",
                "deploy_recipe": "app_host_ppsa_auto_with_locks"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": True,
                "deploy_recipe": "app_host_ppsa"
            }
        },
        "VIDEO": {
            "prod": {
                "release": True,
                "testing_release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "enrich": True,
                "run_smoke_test": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto_with_locks"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": True,
                "deploy_recipe": "app_host_ppsa"
            }
        },
        "NEWS": {
            "prod": {
                "release": True,
                "release_to": "stable",
                "auto_deploy": False,
                "deploy_recipe": "app_host_ppsa"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "deploy_recipe": "app_host_ppsa"
            }
        },
        "COMMON": {
            "prod": {
                "release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "run_smoke_test": True,
                "enrich": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": False,
                "deploy_recipe": "app_host_ppsa_auto"
            }
        },
        "MAIL": {
            "prod": {
                "release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "run_smoke_test": True,
                "enrich": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": False,
                "deploy_recipe": "app_host_ppsa_auto"
            }
        },
        "MAILCORP": {
            "prod": {
                "release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "run_smoke_test": True,
                "enrich": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": True,
                "deploy_recipe": "app_host_ppsa_auto"
            }
        },
        "SHARED": {
            "prod": {
                "release": True,
                "release_to": "stable",
                "auto_deploy": True,
                "svn_check": False,
                "run_smoke_test": True,
                "enrich": True,
                "delete_graphs": True,
                "deploy_recipe": "app_host_ppsa_auto"
            },
            "dev": {
                "release": True,
                "release_to": "unstable",
                "auto_deploy": False,
                "svn_check": True,
                "deploy_recipe": "app_host_ppsa_auto"
            }
        }
    }

    @staticmethod
    def is_work_day(date, oauth_token):
        if oauth_token:
            holidays = CalendarAPI(oauth_token=oauth_token).get_holidays(date, date)
            if date in holidays:
                logging.info("%r is a holiday: %r", date, holidays[date])
            return date not in holidays
        date = datetime.datetime.strptime(date, DATE_FORMAT)
        return date.isoweekday() not in (6, 7)

    def on_execute(self):
        for vertical in get_verticals(self.ctx):
            key = "{}_config_bundle_task_id".format(vertical)
            logging.debug(self.deploy_settings.get(vertical))
            dashboard_key = "{}_deploy_dashboard_task_id".format(vertical)
            settings = self.deploy_settings.get(vertical).get(self.ctx.get(DeployVariant.name))

            calendar_token = self.get_vault_data("APP_HOST", "calendar_token")
            current_date = datetime.datetime.now().strftime(DATE_FORMAT)
            self.ctx[self.IS_WORK_DAY] = self.is_work_day(current_date, calendar_token)
            if not self.ctx[self.IS_WORK_DAY] and settings.get("only_workdays", True):
                self.descr += "{}{} (non-working day)".format(self.DESCRIPTION_SEPARATOR, current_date)
                return

            graph_generator = GraphGenerator.get_resource_from_ctx(self.ctx)

            logging.debug("settings {}".format(settings))
            logging.debug("graph_generator {}".format(graph_generator))

            if not self.ctx.get(key):
                if not settings.get("svn_check", False):
                    resource_type = BuildAppHostConfigBundle.resource_map.get(vertical)

                    releases = channel.sandbox.list_releases(resource_type=resource_type,
                                                             release_status=settings.get('release_to'),
                                                             limit=10,
                                                             order_by="-release__creation_time")
                    tasks = []
                    for release in releases:
                        tasks.append(channel.sandbox.get_task(release.task_id))

                    task_urls = []
                    task_urls_map = {}
                    for task in tasks:
                        url = task.ctx.get(SandboxArcadiaUrlParameter.name)
                        arcadia_url = Arcadia.parse_url(url)
                        task_urls_map[arcadia_url] = url
                        task_urls.append(arcadia_url)

                    task_urls.sort(key=lambda x: x.revision, reverse=True)
                    task_url = task_urls[0]

                    logging.debug("releases")
                    logging.debug(releases)
                    logging.debug("tasks")
                    logging.debug(tasks)
                    logging.debug("task_urls")
                    logging.debug(task_urls)
                    logging.debug(task_url[0])
                    logging.debug(task_urls_map.get(task_url))

                    svn_url = self.ctx.get(SandboxArcadiaUrlParameter.name) or task_urls_map.get(task_url)

                    context = {
                        ChooseVertical.name: self.ctx.get(ChooseVertical.name),
                        GraphGenerator.name: graph_generator,
                        SandboxArcadiaUrlParameter.name: svn_url,
                        BuildOnlyServiceResources.name: False,
                        BuildServiceResources.name: False,
                        BuildTemplatesDir.name: False,
                        EnrichGraphsWithRevisions.name: settings.get("enrich", False),
                        RunSmokeTest.name: settings.get("run_smoke_test", False),
                        AllowGraphRemoval.name: settings.get("delete_graphs", False)
                    }
                    create_task = True
                else:
                    save_run_info(self.ctx)
                    path = '/web/app_host/conf/graph_generator/vertical/{}'.format(vertical)
                    create_task = has_changes_since_prev_run(self.ctx, AutoBuildAppHostConfigBundle.type, path)

                    context = {
                        ChooseVertical.name: self.ctx.get(ChooseVertical.name),
                        GraphGenerator.name: graph_generator,
                        SandboxArcadiaUrlParameter.name: self.ctx.get(SandboxArcadiaUrlParameter.name),
                        BuildOnlyServiceResources.name: False,
                        BuildServiceResources.name: False,
                        BuildTemplatesDir.name: False,
                        EnrichGraphsWithRevisions.name: settings.get("enrich", False),
                        RunSmokeTest.name: settings.get("run_smoke_test", False),
                        AllowGraphRemoval.name: settings.get("delete_graphs", False)
                    }

                if create_task:
                    context.update({
                        NoBackEndsInGraph.name: self.ctx.get(NoBackEndsInGraph.name)
                    })

                    subtask = self.create_subtask(
                        task_type="BUILD_APP_HOST_CONFIG_BUNDLE",
                        description=self.descr,
                        input_parameters=context,
                        priority=self.priority,
                        inherit_notifications=True
                    )
                    self.ctx[key] = subtask.id
                    logging.info(
                        "Starting subtask BUILD_APP_HOST_CONFIG_BUNDLE {}".format(
                            subtask
                        )
                    )
                    # XXX throws WaitTask exception, most probably this task won't work for multiple verticals at the
                    # same time because of this
                    self.wait_tasks(
                        subtask,
                        (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                         self.Status.EXCEPTION, self.Status.TIMEOUT, self.Status.NOT_RELEASED),
                        True,
                        None
                    )
            else:
                if not self.ctx.get(dashboard_key):
                    task_id = self.ctx.get(key)

                    task = channel.sandbox.get_task(task_id)
                    if task.status != "FINISHED":
                        raise SandboxTaskFailureError("Failed to build graphs")

                    if settings.get("release"):

                        if settings.get("testing_release", False) and not self.ctx.get("released_to_testing", False):
                            self.create_release(task_id, status="testing", subject="{}".format(self.descr))
                            self.ctx["released_to_testing"] = True
                            self.wait_tasks(
                                task_id,
                                (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                                 self.Status.EXCEPTION, self.Status.TIMEOUT, self.Status.NOT_RELEASED),
                                True,
                                None
                            )

                        self.create_release(task_id, status=settings.get('release_to'), subject="{}".format(self.descr))

                        if settings.get("auto_deploy"):
                            context = {
                                ReleaseTask.name: task_id,
                                NannyDashboardName.name: get_dashboard_name(vertical),
                                NannyDashboardRecipeName.name: settings.get("deploy_recipe"),
                                NannyDashboardFilter.name: 'stable-app_host',
                                SandboxReleaseType.name: settings.get('release_to'),
                                VaultName.name: 'nanny_oauth_token',
                                NannyWaitDeployParameter.name: True,
                                VaultOwner.name: settings.get("vault_owner", "APP_HOST")
                            }

                            subtask = self.create_subtask(
                                task_type="DEPLOY_NANNY_DASHBOARD",
                                description=self.descr,
                                input_parameters=context,
                                priority=self.priority,
                                inherit_notifications=True
                            )
                            self.ctx[dashboard_key] = subtask.id

                            logging.info(
                                "Starting subtask DEPLOY_NANNY_DASHBOARD {}".format(
                                    subtask
                                )
                            )
                            self.wait_tasks(
                                subtask,
                                (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                                 self.Status.EXCEPTION, self.Status.STOPPED, self.Status.TIMEOUT, self.Status.NOT_RELEASED),
                                True,
                                None
                            )
                else:
                    task = channel.sandbox.get_task(self.ctx.get(dashboard_key))
                    if task.status != "FINISHED":
                        raise SandboxTaskFailureError("Failed to deploy")


__Task__ = AutoBuildAppHostConfigBundle
