# -*- coding: utf-8 -*-

import copy
import json
import logging

from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.common.BaseTestTask import BaseDolbiloTask
from sandbox.projects.common.app_host import AppHostProvider
from sandbox.projects.common.app_host import HttpAdapterProvider
from sandbox.projects.common.app_host import PerfTestServantProvider
from sandbox.projects.common.app_host import CustomProvider
from sandbox.projects.common.app_host import SportProxyProvider
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.network import is_port_free


APPS_TYPE = {
    "app_host": AppHostProvider,
    "perftest_servant": PerfTestServantProvider,
    "http_adapter": HttpAdapterProvider,
    "custom": CustomProvider,
    "sport_proxy": SportProxyProvider,
}

APPS_PORTS_TYPE = {
    "app_host": {"ports": ["main", "grpc", "grpcs"], "sequential": True},
    "perftest_servant": {"ports": ["main", "grpc"], "sequential": False},
    "http_adapter": {"ports": ["main"], "sequential": False},
    "custom": {"ports": ["http", "main", "grpc"], "sequential": True},
    "sport_proxy": {"ports": ["http", "main", "grpc"], "sequential": True},
}

NEXT_PORT = 15001
MAX_PORT = 20000


class BenchmarkConfig(parameters.ResourceSelector):
    name = 'APP_HOST_BENCHMARK_CONFIG'
    description = 'Benchmark config'
    resource_type = resource_types.APP_HOST_BENCHMARK_CONFIG
    attrs = {"type": "benchmark"}


class BenchmarkPlan(parameters.ResourceSelector):
    name = 'APP_HOST_BENCHMARK_PLAN'
    description = 'Dolbilo plan'
    resource_type = resource_types.APP_HOST_BENCHMARK_PLAN


class UseGPerfToolsParameter(parameters.SandboxBoolParameter):
    name = 'use_gperftools'
    description = 'Use gperf cpu profiler'
    default_value = False


class StoreUnistatParameter(parameters.SandboxBoolParameter):
    name = 'store_unistat'
    description = 'Store unistat'
    default_value = True


class AppToShoot(parameters.SandboxSelectParameter):
    name = 'app_to_shoot'
    description = 'which app we should shoot'
    default_value = "app_host"
    choices = [
        ("apphost", "app_host main"),
        ("http_adapter", "http_adapter main")
    ]


class PortToShoot(parameters.SandboxSelectParameter):
    name = 'port_to_shoot'
    description = 'which port we should shoot'
    default_value = "main"
    choices = [
        ("main", "main (neh/http)"),
        ("nehs", "neh secure (neh/http)"),
        ("grpc", "grpc"),
        ("grpcs", "grpc secure"),
    ]


class BenchmarkAppHost(BaseDolbiloTask):
    type = "BENCHMARK_APP_HOST"

    execution_space = 102400
    default_cpu_model = None

    input_parameters = [BenchmarkConfig, BenchmarkPlan, AppToShoot, PortToShoot, UseGPerfToolsParameter,
                        StoreUnistatParameter] + BaseDolbiloTask.input_parameters

    environment = (environments.PipEnvironment('numpy', use_wheel=True),)

    client_tags = Tag.INTEL_E5_2660V1

    config = {}
    apps = {}

    def initCtx(self):
        self.ctx.update(
            {
                'requests_limit': 60000,
                'total_sessions': 1,
                'executor_mode': 'finger',
                'fuckup_mode_max_simultaneous_requests': 1,
                'plan_mode_delay_between_requests_multipliers': 1.0,
            }
        )

    def _start_searcher(self, callbacks=None, **kvargs):
        logging.info('current pid %s' % [self._get_app_to_shoot().process.pid])
        if not self._get_app_to_shoot().alive():
            raise SandboxTaskFailureError('{} is dead'.format(self.ctx.get(AppToShoot.name)))
        return [self._get_app_to_shoot().process]

    def _stop_searcher(self, subprocess_list, session_id=None, **kvargs):
        pass

    def _get_app_to_shoot(self):
        logging.debug(self.apps)
        logging.debug(AppToShoot.name)
        logging.debug(self.ctx.get(AppToShoot.name))
        return self.apps.get(self.ctx.get(AppToShoot.name, "app_host"))

    def _get_searcher_port(self):
        return self._get_app_to_shoot().port

    @staticmethod
    def _get_port(batch=1):
        global NEXT_PORT
        port = NEXT_PORT
        while not all(is_port_free(port + i) for i in xrange(batch)):
            port = port + 1
        NEXT_PORT = port + batch
        if NEXT_PORT > MAX_PORT:
            raise ValueError("app host benchmark is out of ports")
        return port

    def _get_ports(self, marks=None):
        if marks is None:
            marks = {"ports": ["main"], "sequential": True}

        result = {}
        if marks["sequential"]:
            port = self._get_port(len(marks["ports"]))
            for key in marks["ports"]:
                result[key] = port
                port = port + 1
        else:
            for key in marks["ports"]:
                result[key] = self._get_port()

        return result

    def setup(self):
        config_path = self.sync_resource(self.ctx[BenchmarkConfig.name])
        with open(config_path, 'r') as fh:
            self.config = json.load(fh)
        logging.debug('benchmark config {}'.format(str(self.config)))

        self.apps = {}
        self.ctx["app_ports"] = {}
        self.ctx['abs_path'] = self.abs_path()

        APPS_MAP = [(k,  # app_name
                     v.get('type', False),  # app_type
                     APPS_TYPE.get(v.get('type'), False),  # app_class
                     v,  # app_config
                     self._get_ports(APPS_PORTS_TYPE.get(v.get('type')))  # app_ports
                     ) for k, v in self.config.iteritems()
                    ]
        logging.debug(APPS_MAP)
        for app_name, app_type, app_class, app_config, app_port in APPS_MAP:
            app = app_class()
            app.options = app_config.get('options')
            app.resources = app_config.get('resources')
            app.use_profiler = self.ctx[UseGPerfToolsParameter.name]
            app.use_gperftools = self.ctx[UseGPerfToolsParameter.name]
            app.save_unistat = self.ctx[StoreUnistatParameter.name]
            app.human_name = app_name
            app.ports = app_port

            for proto, port in app_port.iteritems():
                self.ctx.get("app_ports").setdefault(app_type, []).append((app_name, proto, app_port.get(proto)))

            self.apps[app_name] = app

        self._get_app_to_shoot().protocol_for_backends = self.ctx.get(PortToShoot.name, "main")

        for app_name, app_object in self.apps.iteritems():
            app_object.ctx = copy.deepcopy(self.ctx)
            app_object.prepare()
            app_object.start()

    def benchmark(self):
        self.ctx['requests_per_sec'] = []

        params = copy.deepcopy(self.ctx)
        params['requests_per_sec'] = []
        self.run_dolbilo(params, 'benchmark', [], start_once=True)
        self.ctx['requests_per_sec'] = \
            self.ctx.get('requests_per_sec', []) + params['requests_per_sec']

    def on_execute(self):
        logging.info('ctx# ' + str(self.ctx))
        self.ctx['dolbilo_plan_resource_id'] = self.ctx[BenchmarkPlan.name]
        self.config = None

        with self.current_action('setup'):
            self.setup()

        with self.current_action('benchmarking'):
            self.benchmark()

        with self.current_action('saving resources'):
            for app_name, app in self.apps.iteritems():
                app.save()

        with self.current_action('combining resources'):
            pass

        with self.current_action('teardown'):
            for app_name, app in self.apps.iteritems():
                if app.alive():
                    app.stop()

        logging.info('ctx# ' + str(self.ctx))
