import copy
import logging
import string
import random

from sandbox.projects.common.app_host import Benchmark
from sandbox.projects.common.app_host import BASE_BENCHMARK_CONFIG
from sandbox.projects.resource_types import APP_HOST_BENCHMARK_PLAN, OTHER_RESOURCE

from sandbox.sandboxsdk.process import run_process

_BENCHMARK_QUOTA_FACTOR = 5


class SynthPointGraphBenchmark(Benchmark):
    def __init__(
            self,
            desc='graph with single node',
            requests_limit=100000,
            resp_size_distribution='1024,0,0',
            resp_time_distribution='0,0,0',
            trash_prob=0.0,
            structured_trash_prob=0.0,
            consistency_check_aware_trash_prob=0.0,
            break_some_bytes_prob=0.0,
            broken_byte_count=1,
            concurrency=1,
            source_codecs=None,
            plan_id=None,
            store_unistat=True,
            use_grpc=False,
            use_encryption=False,
            resources=None):
        self.desc = desc

        self.dolbilo_plan_id = plan_id
        self.executor_mode = 'finger'
        self.fuckup_mode_max_simultaneous_requests = concurrency
        self.total_sessions = 1
        self.requests_limit = requests_limit

        self.options = dict()
        self.options['concurrency'] = concurrency
        self.options['resp_size_distribution'] = resp_size_distribution
        self.options['resp_time_distribution'] = resp_time_distribution
        self.options['trash_prob'] = trash_prob
        self.options['structured_trash_prob'] = structured_trash_prob
        self.options['consistency_check_aware_trash_prob'] = \
            consistency_check_aware_trash_prob
        self.options['break_some_bytes_prob'] = break_some_bytes_prob
        self.options['broken_byte_count'] = broken_byte_count
        self.options['source_codecs'] = source_codecs
        self.options['use_grpc'] = use_grpc
        self.use_grpc = use_grpc
        self.use_encryption = use_encryption
        self.resources = resources
        self.store_unistat = store_unistat

    def setup(self, parent_task):
        cfg = copy.deepcopy(BASE_BENCHMARK_CONFIG)

        apphost_cfg = cfg.get('app_host').get('options').get('config')
        apphost_cfg['threads'] = self.options.get('concurrency')
        apphost_cfg['total_quota'] = _BENCHMARK_QUOTA_FACTOR * self.options.get('concurrency')
        if self.options.get('source_codecs') is not None:
            apphost_cfg['source_codecs'] = self.options.get('source_codecs')

        graph = cfg.get('app_host').get('options').get('graphs').get('shoot.json').get('sources').get('SOURCE_1')

        if self.use_grpc:
            graph["use_grpc"] = self.use_grpc

        if self.use_encryption:
            graph["use_encryption"] = self.use_encryption

        cfg.get('perftest').get('options')['threads'] = self.options.get('concurrency')
        cfg.get('perftest').get('options')['use_grpc'] = self.use_grpc
        for option in ['trash_prob', 'structured_trash_prob', 'broken_byte_count',
                       'consistency_check_aware_trash_prob', 'break_some_bytes_prob',
                       'resp_size_distribution', 'resp_time_distribution']:
            cfg.get('perftest').get('options')[option] = self.options.get(option)

        cfg['perftest_servant'] = copy.deepcopy(cfg.get('perftest'))

        cfg = self.clean_template(cfg)

        logging.debug(cfg)
        cfg = self.add_resources(cfg)

        self.create_config_resource(cfg, parent_task)


class WideGraphBenchmark(Benchmark):
    def __init__(
            self,
            desc='wide graph',
            requests_limit=100000,
            resp_size_distribution='1024,0,0',
            resp_time_distribution='0,0,0',
            concurrency=1,
            breadth=4,
            source_codecs=None,
            plan_id=None,
            store_unistat=True,
            use_grpc=False,
            use_encryption=False,
            resources=None):
        self.desc = desc

        self.dolbilo_plan_id = plan_id

        self.concurrency = concurrency
        self.executor_mode = 'finger'
        self.fuckup_mode_max_simultaneous_requests = concurrency
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.resp_size_distribution = resp_size_distribution
        self.resp_time_distribution = resp_time_distribution
        self.source_codecs = source_codecs
        self.breadth = breadth
        self.store_unistat = store_unistat
        self.use_grpc = use_grpc
        self.use_encryption = use_encryption
        self.resources = resources

    def setup(self, parent_task):
        cfg = copy.deepcopy(BASE_BENCHMARK_CONFIG)

        apphost_cfg = cfg.get('app_host').get('options').get('config')
        apphost_cfg['threads'] = self.concurrency
        apphost_cfg['total_quota'] = _BENCHMARK_QUOTA_FACTOR * self.concurrency
        if self.source_codecs is not None:
            apphost_cfg['source_codecs'] = self.source_codecs

        # describe graph
        graph = {}
        for x in xrange(0, self.breadth):
            graph['perftest-{}'.format(x)] = ['INIT']
        deps = list(graph.keys())
        graph['RESPONSE'] = ['INIT'] + deps
        cfg.get('app_host').get('options').get('graphs').get('shoot.json')['graph'] = graph

        # describe sources
        sources = {}
        for x in xrange(0, self.breadth):
            sources['perftest-{}'.format(x)] = {
                'backend_config': {
                    'backend_descrs': [
                        {
                            'ip': '127.0.0.1',
                            'host': 'localhost',
                            'protocol': 'post',
                            'port': 13400 + x,
                            'weight': 1.0
                        }
                    ]
                },
                'use_grpc': self.use_grpc,
                "use_encryption": self.use_encryption,
                'timeout': 500
            }

        cfg.get('app_host').get('options').get('graphs').get('shoot.json')['sources'] = sources
        # multiply perftest servants
        perftest = copy.deepcopy(cfg.get('perftest'))
        perftest.get('options')['threads'] = self.concurrency
        perftest.get('options')['resp_size_distribution'] = self.resp_size_distribution
        perftest.get('options')['resp_time_distribution'] = self.resp_time_distribution
        perftest.get('options')['use_grpc'] = self.use_grpc
        perftest.get('options')['use_encryption'] = self.use_encryption

        for x in xrange(0, self.breadth):
            pt = copy.deepcopy(perftest)
            port = pt.get('options').get('port')
            pt.get('options')['port'] = port + x
            cfg['perftest-{}'.format(x)] = pt

        cfg = self.clean_template(cfg)

        cfg = self.add_resources(cfg)

        self.create_config_resource(cfg, parent_task)


class BalancingBenchmark(Benchmark):
    def __init__(
            self,
            desc='balancing graph',
            requests_limit=100000,
            perftest_servants=None,
            concurrency=1,
            timeout=10,
            plan_id=None,
            source_codecs='null',
            balancing_scheme='urandom',
            store_unistat=True,
            use_grpc=False,
            use_encryption=False,
            resources=None):
        self.desc = desc

        self.dolbilo_plan_id = plan_id

        self.concurrency = concurrency
        self.executor_mode = 'finger'
        self.fuckup_mode_max_simultaneous_requests = concurrency
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.source_codecs = source_codecs
        self.perftest_servants = perftest_servants or [(True, 1, '1024,0,0', '0,0,0', 0)]
        self.balancing_scheme = balancing_scheme
        self.timeout = timeout
        self.store_unistat = store_unistat
        self.use_grpc = use_grpc
        self.use_encryption = use_encryption
        self.resources = resources

    def setup(self, parent_task):
        cfg = copy.deepcopy(BASE_BENCHMARK_CONFIG)

        apphost_cfg = cfg.get('app_host').get('options').get('config')
        apphost_cfg['threads'] = self.concurrency
        apphost_cfg['total_quota'] = _BENCHMARK_QUOTA_FACTOR * self.concurrency
        if self.source_codecs is not None:
            apphost_cfg['source_codecs'] = self.source_codecs

        # describe sources
        counter = 0
        backends = []
        perftest = copy.deepcopy(cfg.get('perftest'))
        perftest.get('options')['threads'] = self.concurrency
        del cfg['perftest']
        for servant in self.perftest_servants:
            active, threads, rs, rt, tp = servant
            backends.append(
                {
                    "ip": "127.0.0.1",
                    "host": "localhost",
                    "protocol": "post",
                    "port": 13400 + counter,
                    "weight": 1.0
                }
            )
            pt = copy.deepcopy(perftest)
            pt.get('options')['threads'] = threads
            pt.get('options')['port'] = 13400 + counter
            pt.get('options')['resp_size_distribution'] = rs
            pt.get('options')['resp_time_distribution'] = rt
            pt.get('options')['trash_prob'] = tp
            pt.get('options')['active'] = active
            pt.get('options')['use_grpc'] = self.use_grpc
            pt.get('options')['use_encryption'] = self.use_encryption
            cfg['perftest-{}'.format(counter)] = pt
            counter += 1

        cfg.get('app_host').get('options').get('graphs').get('shoot.json')['sources'] = \
            {
                "SOURCE_1": {
                    "backend_config": {
                        "backend_descrs": backends,
                    },
                    "use_grpc": self.use_grpc,
                    "use_encryption": self.use_encryption,
                    "balancing_scheme": self.balancing_scheme,
                    "timeout": self.timeout
                }
            }

        cfg = self.clean_template(cfg)

        cfg = self.add_resources(cfg)

        self.create_config_resource(cfg, parent_task)


class RequestFractionBenchmark(Benchmark):
    # see APPHOST-101

    def __init__(
            self,
            desc='request fraction benchmark',
            requests_limit=200000,
            main_servant=(True, 1, '1024,0,0', '0,0,0', 0),
            test_servant=(True, 1, '1024,0,0', '10,0,0', 0),
            concurrency=1,
            timeout=10,
            plan_id=None,
            store_unistat=True,
            use_grpc=True,
            use_encryption=False,
            resources=None):
        self.desc = desc

        self.dolbilo_plan_id = plan_id

        self.concurrency = concurrency
        self.executor_mode = 'finger'
        self.fuckup_mode_max_simultaneous_requests = concurrency
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.main_servant = main_servant
        self.test_servant = test_servant
        self.timeout = timeout
        self.store_unistat = store_unistat
        self.use_grpc = use_grpc
        self.use_encryption = use_encryption
        self.resources = resources

    def setup(self, parent_task):
        cfg = copy.deepcopy(BASE_BENCHMARK_CONFIG)

        logging.info("config: {}".format(cfg))

        apphost_cfg = cfg.get('app_host').get('options').get('config')
        apphost_cfg['threads'] = self.concurrency
        apphost_cfg['total_quota'] = _BENCHMARK_QUOTA_FACTOR * self.concurrency

        # describe sources
        perftest = copy.deepcopy(cfg.get('perftest'))
        perftest.get('options')['threads'] = self.concurrency
        del cfg['perftest']

        # describe main source
        active, threads, rs, rt, tp = self.main_servant
        main_src = copy.deepcopy(perftest)
        main_src.get("options")["threads"] = threads
        main_src.get('options')['port'] = 13400
        main_src.get('options')['resp_size_distribution'] = rs
        main_src.get('options')['resp_time_distribution'] = rt
        main_src.get('options')['trash_prob'] = tp
        main_src.get('options')['active'] = active
        main_src.get('options')['use_grpc'] = self.use_grpc
        main_src.get('options')['use_encryption'] = self.use_encryption

        cfg['MAIN_SOURCE'] = main_src
        cfg.get('app_host').get('options').get('graphs').get('shoot.json')['sources'] = \
            {
                "MAIN_SOURCE": {
                    "backend_config": {
                        "backend_descrs": [
                            {
                                "ip": "127.0.0.1",
                                "host": "localhost",
                                "protocol": "post",
                                "port": main_src.get("options").get("port"),
                                "weight": 1.0
                            }
                        ]
                    },
                    "use_encryption": self.use_encryption,
                    "use_grpc": self.use_grpc,
                    "balancing_scheme": "urandom",
                    "timeout": self.timeout
                }
            }
        graph = {
            "RESPONSE": ["MAIN_SOURCE"],
            "MAIN_SOURCE": ["INIT"]
        }
        # describe test source
        if self.test_servant is not None:
            active, threads, rs, rt, tp = self.test_servant
            if active:
                test_src = copy.deepcopy(perftest)
                test_src.get("options")["threads"] = threads
                test_src.get('options')['port'] = 13401
                test_src.get('options')['resp_size_distribution'] = rs
                test_src.get('options')['resp_time_distribution'] = rt
                test_src.get('options')['trash_prob'] = tp
                test_src.get('options')['active'] = active
                cfg['TEST_SOURCE'] = test_src
                cfg.get('app_host') \
                    .get('options') \
                    .get('graphs') \
                    .get('shoot.json') \
                    .get('sources')["TEST_SOURCE"] = \
                    {
                        "backend_config": {
                            "backend_descrs": [
                                {
                                    "ip": "127.0.0.1",
                                    "host": "localhost",
                                    "protocol": "post",
                                    "port": test_src.get("options").get("port"),
                                    "weight": 1.0
                                }
                            ]
                        },
                        "use_encryption": self.use_encryption,
                        "use_grpc": self.use_grpc,
                        "balancing_scheme": "urandom",
                        "request_fraction": 1.0,
                        "timeout": self.timeout
                    }
                graph["TEST_SOURCE"] = ["INIT"]
        cfg.get("app_host").get('options').get("graphs").get("shoot.json")["graph"] = graph

        cfg = self.clean_template(cfg)

        cfg = self.add_resources(cfg)

        self.create_config_resource(cfg, parent_task)


class StreamingBenchmark(Benchmark):
    def __init__(
            self,
            desc='Streaming benchmark',
            requests_limit=100000,
            resp_size_distribution='1024,0,0',
            resp_time_distribution='0,0,0',
            plan_id=None,
            store_unistat=True,
            use_grpc=False,
            use_encryption=False,
            resources=None):
        self.desc = desc

        self.dolbilo_plan_id = plan_id

        self.executor_mode = 'finger'
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.resp_size_distribution = resp_size_distribution
        self.resp_time_distribution = resp_time_distribution
        self.store_unistat = store_unistat
        self.use_grpc = use_grpc
        self.use_encryption = use_encryption
        self.resources = resources

    def setup(self, parent_task):
        template = copy.deepcopy(BASE_BENCHMARK_CONFIG)
        graph = template.get("app_host").get("options").get("graphs").get('shoot.json')
        graph["graph"] = {
            "RESPONSE": [
                "STREAMING_SOURCE"
            ],
            "FIRST_DEPENDENCY": [
                "INIT"
            ],
            "SECOND_DEPENDENCY": [
                "FIRST_DEPENDENCY",
                "INIT"
            ],
            "THIRD_DEPENDENCY": [
                "SECOND_DEPENDENCY",
                "INIT"
            ],
            "STREAMING_SOURCE": [
                "FIRST_DEPENDENCY",
                "SECOND_DEPENDENCY",
                "THIRD_DEPENDENCY"
            ]
        }
        graph["sources"] = {
            "FIRST_DEPENDENCY": {
                "timeout": 1000,
                "use_grpc": self.use_grpc
            },
            "SECOND_DEPENDENCY": {
                "timeout": 1000,
                "use_grpc": self.use_grpc
            },
            "THIRD_DEPENDENCY": {
                "timeout": 1000,
                "use_grpc": self.use_grpc
            },
            "STREAMING_SOURCE": {
                "timeout": 1000,
                "use_grpc": self.use_grpc,
                "use_streaming": True
            },
        }

        perftest_servant = copy.deepcopy(template.get("perftest"))
        perftest_servant.get("options")["use_grpc"] = self.use_grpc

        for dep in ["FIRST_DEPENDENCY", "SECOND_DEPENDENCY", "THIRD_DEPENDENCY"]:
            template[dep] = copy.deepcopy(perftest_servant)

        streaming_source = copy.deepcopy(perftest_servant)
        streaming_source.get("options")["custom_options"] = " --continue-on-empty-deps"

        template["STREAMING_SOURCE"] = streaming_source

        template = self.clean_template(template)

        template = self.add_resources(template)

        self.create_config_resource(template, parent_task)


class SubhostBenchmark(Benchmark):
    def __init__(
            self,
            desc='Streaming benchmark',
            requests_limit=100000,
            resp_size_distribution='1024,0,0',
            resp_time_distribution='0,0,0',
            plan_id=None,
            store_unistat=True,
            use_grpc=False,
            use_subhost_streaming=False,
            use_encryption=False,
            resources=None):
        self.desc = desc

        self.dolbilo_plan_id = plan_id

        self.executor_mode = 'finger'
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.resp_size_distribution = resp_size_distribution
        self.resp_time_distribution = resp_time_distribution
        self.store_unistat = store_unistat
        self.use_grpc = use_grpc
        self.use_subhost_streaming = use_subhost_streaming
        self.use_encryption = use_encryption
        self.resources = resources

    def setup(self, parent_task):
        template = copy.deepcopy(BASE_BENCHMARK_CONFIG)
        graph = template.get("app_host").get("options").get("graphs").get('shoot.json')
        subgraph = copy.deepcopy(template.get("app_host").get("options").get("graphs").get('shoot.json'))

        graph["graph"] = {
            "RESPONSE": [
                "SUBHOST"
            ],
            "SUBHOST": [
                "INIT"
            ]
        }
        graph["sources"] = {
            "SUBHOST": {
                "timeout": 1000,
                "backend_config": {
                    "backend_descrs": [
                        {
                            "host": "localhost",
                            "protocol": "inproc",
                            "path": "/_subhost/subgraph",
                            "min_weight": 0.02,
                            "weight": 0.5,
                            "port": 1
                        }
                    ]
                }
            }
        }

        subgraph["graph"] = {
            "FIRST_RESPONSE": [
                "FIRST_DEPENDENCY"
            ],
            "SECOND_RESPONSE": [
                "SECOND_DEPENDENCY"
            ],
            "THIRD_RESPONSE": [
                "THIRD_DEPENDENCY"
            ],
            "FIRST_DEPENDENCY": [
                "INIT"
            ],
            "SECOND_DEPENDENCY": [
                "FIRST_DEPENDENCY",
                "INIT"
            ],
            "THIRD_DEPENDENCY": [
                "SECOND_DEPENDENCY",
                "INIT"
            ]
        }

        subgraph["output_deps"] = ["FIRST_RESPONSE", "SECOND_RESPONSE", "THIRD_RESPONSE"]

        subgraph["sources"] = {
            "FIRST_DEPENDENCY": {
                "timeout": 1000,
                "use_grpc": self.use_grpc
            },
            "SECOND_DEPENDENCY": {
                "timeout": 1000,
                "use_grpc": self.use_grpc
            },
            "THIRD_DEPENDENCY": {
                "timeout": 1000,
                "use_grpc": self.use_grpc
            }
        }

        template.get("app_host").get("options").get("graphs")["subgraph.json"] = subgraph

        perftest_servant = copy.deepcopy(template.get("perftest"))
        perftest_servant.get("options")["use_grpc"] = self.use_grpc

        for dep in ["FIRST_DEPENDENCY", "SECOND_DEPENDENCY", "THIRD_DEPENDENCY"]:
            template[dep] = copy.deepcopy(perftest_servant)

        template = self.clean_template(template)

        template = self.add_resources(template)

        self.create_config_resource(template, parent_task)


class CustomBenchmark(Benchmark):
    def __init__(
            self,
            desc='custom benchmark',
            requests_limit=100000,
            concurrency=1,
            json_contexts_id=None,
            dplanner_executable_id=None,
            make_tank_ammo_executable_id=None,
            ctx=None,
            resources=None,
            setup_path=None,
            source_timeout=150,
            metrics=None):

        self.desc = desc
        self.dolbilo_plan_id = None
        self.json_contexts_id = json_contexts_id
        self.dplanner_executable_id = dplanner_executable_id
        self.make_tank_ammo_executable_id = make_tank_ammo_executable_id

        self.concurrency = concurrency
        self.executor_mode = 'finger'
        self.fuckup_mode_max_simultaneous_requests = self.concurrency
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.resources = resources
        self.ctx = ctx
        self.store_unistat = True
        self.setup_path = setup_path
        self.source_timeout = source_timeout
        self.metrics = []
        if metrics:
            self.metrics = [m.rstrip().lstrip() for m in metrics.split(",")]

    def setup(self, parent_task):
        template = copy.deepcopy(BASE_BENCHMARK_CONFIG)

        backend = {
            "ip": "127.0.0.1",
            "host": "localhost",
            "protocol": "post",
            "port": 13400,
            "weight": 1.0
        }
        if self.setup_path:
            backend["path"] = self.setup_path

        graph = {
            "sources": {
                "CUSTOM": {
                    "backend_config": {
                        "backend_descrs": [backend]
                    },
                    "can_process_subgraph": 0,
                    "logging_config": {
                        "dump_request_probability": 0.1
                    },
                    "timeout": self.source_timeout,
                    "balancing_scheme": "weighted",
                    "path": self.setup_path
                }
            },
            "graph": {
                "CUSTOM": [
                    "INIT"
                ],
                "RESPONSE": [
                    "CUSTOM"
                ]
            },
            "input_deps": [
                "INIT"
            ]
        }

        template.get('app_host').get('options')['graphs'] = {}
        template.get('app_host').get('options').get('graphs')['custom.json'] = graph

        template.get('app_host').get('options').get('config')['threads'] = self.concurrency

        template["CUSTOM"] = {
            "type": "custom",
            "options": {
                "metrics": self.metrics
            }
        }

        template = self.clean_template(template)

        logging.debug(template)

        template = self.add_resources(template)

        self.create_config_resource(template, parent_task)

    def prepare_plan(self, parent_task):
        def _prepare_post(context):
            if context is dict:
                init_context = {"name": "INIT"}
                for source in context.get("answers"):
                    for k, v in source.iteritems():
                        if k == "name":
                            continue
                        init_context.get(k, []).extend(v)
                context = [init_context]
            headers = [
                "Host: localhost",
                "Accept-Encoding: gzip, deflate, compress",
                "Accept: */*",
                "Content-Type: application/json; charset=utf-8",
                "User-Agent: BenchmarkAppHost"
            ]
            request = 'POST /_ctx/_json/custom HTTP/1.0\r\n'
            request += '\r\n'.join(headers)
            request += '\r\nContent-Length: ' + str(len(context))
            request += '\r\n\r\n' + context
            request += '\r\n\r\n'
            return str(len(request)) + '\r\n' + request

        def _get_random_name(length):
            return ''.join(random.choice(string.ascii_lowercase) for _ in range(length))

        name = "custom"
        json_contexts = parent_task.sync_resource(self.json_contexts_id)
        dplanner = parent_task.sync_resource(self.dplanner_executable_id)

        contexts_res = []
        with open(json_contexts) as fd:
            contexts = fd.readlines()

        for ctx in contexts:
            contexts_res.append(_prepare_post(ctx))

        random_name = _get_random_name(5)
        phantom_name = "{}.{}.phantom".format(name, random_name)
        dolbilo_name = "{}.{}.dolbilo".format(name, random_name)

        with open(phantom_name, "w") as fp:
            i = 0
            while i < self.requests_limit:
                for ctx in contexts_res:
                    fp.write(ctx)
                    i += 1
                    if i >= self.requests_limit:
                        break

        resource_id = parent_task.create_resource("custom plan phantom {}".format(self.requests_limit),
                                                  phantom_name,
                                                  OTHER_RESOURCE,
                                                  attributes={"app": "phantom"})

        parent_task.mark_resource_ready(resource_id)

        dplanner_cmd = "{dplanner} -l {phantom_name} -o {dolbilo_name} -t phantom"

        run_process(
            dplanner_cmd.format(dplanner=dplanner, phantom_name=phantom_name, dolbilo_name=dolbilo_name).split(),
            shell=True,
            log_prefix=name)

        resource_id = parent_task.create_resource("custom plan {}".format(self.requests_limit),
                                                  dolbilo_name,
                                                  APP_HOST_BENCHMARK_PLAN,
                                                  attributes={"app": "custom"}).id

        parent_task.mark_resource_ready(resource_id)

        self.dolbilo_plan_id = resource_id

        return resource_id


class HttpAdapterBasicBenchmark(Benchmark):
    def __init__(
            self,
            desc='http_adapter benchmark',
            requests_limit=200000,
            concurrency=1,
            timeout=10,
            ctx=None,
            store_unistat=True,
            dplanner_executable_id=None,
            compression=True,
            shoot_to_perf_servants=False,
            resp_size_distribution='1024,0,0',
            resp_time_distribution='0,0,0',
            trash_prob=0.0,
            structured_trash_prob=0.0,
            consistency_check_aware_trash_prob=0.0,
            break_some_bytes_prob=0.0,
            broken_byte_count=1,
            use_grpc=False,
            use_encryption=False,
            resources=None):
        self.desc = desc
        self.dplanner_executable_id = dplanner_executable_id
        self.dolbilo_plan_id = None

        self.concurrency = concurrency
        self.executor_mode = 'finger'
        self.fuckup_mode_max_simultaneous_requests = self.concurrency
        self.total_sessions = 1
        self.requests_limit = requests_limit
        self.timeout = timeout
        self.resources = resources
        self.ctx = ctx
        self.store_unistat = store_unistat
        self.compression = compression
        self.shoot_to_perf_servants = shoot_to_perf_servants
        self.app_to_shoot = "http_adapter"
        self.use_grpc = use_grpc
        self.use_encryption = use_encryption
        self.options = dict()
        self.options['concurrency'] = concurrency
        self.options['resp_size_distribution'] = resp_size_distribution
        self.options['resp_time_distribution'] = resp_time_distribution
        self.options['trash_prob'] = trash_prob
        self.options['structured_trash_prob'] = structured_trash_prob
        self.options['consistency_check_aware_trash_prob'] = \
            consistency_check_aware_trash_prob
        self.options['break_some_bytes_prob'] = break_some_bytes_prob
        self.options['broken_byte_count'] = broken_byte_count

    def setup(self, parent_task):
        template = self.pre_setup(parent_task)

        template = self.add_resources(template)

        self.create_config_resource(template, parent_task)

    def pre_setup(self, parent_task):
        template = copy.deepcopy(BASE_BENCHMARK_CONFIG)

        template["app_host"]["options"]["graphs"]["shoot.json"] = {
            "graph": {
                "RESPONSE": [
                    "HTTP_REQUEST",
                    "SOURCE_1"
                ],
                "SOURCE_1": [
                    "HTTP_REQUEST"
                ]
            },
            "input_deps": ["HTTP_REQUEST"],
            "sources": {
                "SOURCE_1": {
                    "backend_descrs": [
                        {
                            "ip": "127.0.0.1",
                            "host": "localhost",
                            "protocol": "post",
                            "port": 13400,
                            "weight": 1.0
                        }
                    ],
                    "use_grpc": self.use_grpc,
                    "use_encryption": self.use_encryption,
                    "timeout": 500
                }
            }
        }
        app_to_shoot = "app_host"
        if self.shoot_to_perf_servants:
            app_to_shoot = "perftest_servant"

        template["http_adapter"] = {
            "type": "http_adapter",
            "options": {
                "enable_compression": self.compression,
                "app_to_shoot": app_to_shoot
            }
        }

        template["perftest-0"] = copy.deepcopy(template.get('perftest'))

        if self.shoot_to_perf_servants:
            template.get("perftest-0").get("options")["response_type"] = "http_response"
            template.get("perftest-0").get("options")[
                "custom_options"] = "--service-path /shoot --generate-http-response"
            template.get('perftest-0').get("options")["use_grpc"] = self.use_grpc
            template.get('perftest-0').get("options")["use_encryption"] = self.use_encryption
            template.pop("app_host", None)
            for option in ['trash_prob', 'structured_trash_prob', 'broken_byte_count',
                           'consistency_check_aware_trash_prob', 'break_some_bytes_prob',
                           'resp_size_distribution', 'resp_time_distribution']:
                template.get('perftest-0').get('options')[option] = self.options.get(option)

        template = self.clean_template(template)

        logging.debug(template)

        return template

    def prepare_plan(self, parent_task):
        def _prepare_request():
            headers = [
                "Host: localhost",
                "Accept-Encoding: gzip, deflate, compress",
                "Accept: */*",
                "Content-Type: application/json; charset=utf-8",
                "User-Agent: BenchmarkAppHost"
            ]
            request = 'GET /request HTTP/1.0\r\n'
            request += '\r\n'.join(headers)
            request += '\r\n\r\n'
            return str(len(request)) + '\r\n' + request

        def _get_random_name(length):
            return ''.join(random.choice(string.ascii_lowercase) for _ in range(length))

        name = "http_adapter"
        random_name = _get_random_name(5)
        phantom_name = "{}.{}.phantom".format(name, random_name)
        dolbilo_name = "{}.{}.dolbilo".format(name, random_name)
        dplanner = parent_task.sync_resource(self.dplanner_executable_id)

        with open(phantom_name.format(name=name), "w") as fp:
            for i in range(self.requests_limit):
                fp.write(_prepare_request())

        resource_id = parent_task.create_resource("http_adapter plan phantom {}".format(self.requests_limit),
                                                  phantom_name,
                                                  OTHER_RESOURCE,
                                                  attributes={"app": "phantom"})

        parent_task.mark_resource_ready(resource_id)

        dplanner_cmd = "{dplanner} -l {phantom_name} -o {dolbilo_name} -t phantom"

        run_process(
            dplanner_cmd.format(dplanner=dplanner, phantom_name=phantom_name, dolbilo_name=dolbilo_name).split(),
            shell=True,
            log_prefix=name)

        resource_id = parent_task.create_resource("http_adapter plan {}".format(self.requests_limit),
                                                  dolbilo_name,
                                                  APP_HOST_BENCHMARK_PLAN,
                                                  attributes={"app": "http_adapter"}).id

        parent_task.mark_resource_ready(resource_id)

        self.dolbilo_plan_id = resource_id

        return resource_id
