import json

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.resource_types import APP_HOST_BENCHMARK_TESTS
from sandbox.projects.common.app_host.options import AppHostBundle, DplannerExecutable
from sandbox.projects.app_host.BenchmarkAppHostAll import Concurrency, TimeMultiplier
from sandbox.projects.app_host.BenchmarkAppHostAll.benchmarks import CustomBenchmark
from sandbox.projects.app_host.resources import APP_HOST_JSON_CONTEXTS, APP_HOST_TOOL_MAKE_TANK_AMMO_EXECUTABLE


class AppHostJsonContexts(parameters.ResourceSelector):
    name = 'APP_HOST_JSON_CONTEXTS'
    description = 'list of newline separated json contexts'
    resource_type = APP_HOST_JSON_CONTEXTS


class MakeTankAmmo(parameters.ResourceSelector):
    name = 'APP_HOST_TOOL_MAKE_TANK_AMMO_EXECUTABLE'
    description = 'make_tank_ammo'
    resource_type = APP_HOST_TOOL_MAKE_TANK_AMMO_EXECUTABLE


class AnyApplicationArchive(parameters.ResourceSelector):
    name = "any_app_resource"
    description = "Select archive you want to test (though it should meet requiriments)"


class Doc(parameters.SandboxInfoParameter):
    description = """
    How to https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/app_host/BenchmarkAppHostCustom/README
    """


class SourceTimeout(parameters.SandboxIntegerParameter):
    name = "timeout"
    description = "source timeout"
    default_value = 150


class SourcePath(parameters.SandboxStringParameter):
    name = "path"
    description = "Path for source, default None"
    default_value = None


class MetricsToCollect(parameters.SandboxStringParameter):
    name = "metrics"
    description = "Comma separated list of unistat metrics to collect (_golovan handle)"
    default_value = ','.join(['SELF-SS-Requests_dmmm', 'SELF-SS-Successes_dmmm', 'SELF-SS-Failures_dmmm',
                              'SELF-SS-AverageResponseTimeMcs_ammv', 'SELF-SS-ResponseTimesMcsHistogram_dhhh',
                              'SELF-SS-ResponseSizeHistogram_dhhh'])


class BenchmarkAppHostCustomBinary(SandboxTask):
    type = "BENCHMARK_APP_HOST_CUSTOM"

    input_parameters = [Doc, Concurrency, TimeMultiplier, AppHostBundle, DplannerExecutable, MakeTankAmmo,
                        AppHostJsonContexts, AnyApplicationArchive, SourceTimeout, SourcePath, MetricsToCollect]

    def on_create(self):
        self.descr = open("README").read()

    def on_execute(self):
        if not self.ctx.get('children', []):
            concurrency = int(self.ctx[Concurrency.name])
            time_multiplier = float(self.ctx.get(TimeMultiplier.name, 1.0))
            threads = (concurrency + 1) / 2

            benchmark = CustomBenchmark(
                requests_limit=int(1000 * concurrency * time_multiplier),
                desc="Custom Benchmark",
                json_contexts_id=self.ctx.get(AppHostJsonContexts.name),
                dplanner_executable_id=self.ctx.get(DplannerExecutable.name),
                make_tank_ammo_executable_id=self.ctx.get(MakeTankAmmo.name),
                concurrency=threads,
                resources={
                    "app_host": {
                        "type": "app_host",
                        "binary": {"id": self.ctx[AppHostBundle.name], "type": "file"}
                    },
                    "custom": {
                        "type": "custom",
                        "binary": {"id": self.ctx.get(AnyApplicationArchive.name), "type": "file"}
                    }
                },
                metrics=self.ctx.get(MetricsToCollect.name),
                setup_path=self.ctx.get(SourcePath.name),
                source_timeout=self.ctx.get(SourceTimeout.name))

            benchmark.prepare_plan(self)
            bench_id = benchmark.create_task(self)
            self.ctx["children"] = {benchmark.desc: bench_id.id}
            self.wait_tasks(
                bench_id,
                (self.Status.SUCCESS, self.Status.FAILURE, self.Status.DELETED, self.Status.RELEASED,
                 self.Status.EXCEPTION),
                True,
                None
            )
        else:
            for task_descr, task_id in self.ctx.get('children', {}).iteritems():

                task = channel.sandbox.get_task(task_id)
                if task.status != "FINISHED":
                    raise SandboxTaskFailureError("Failed to launch benchmark")

            tests_path = 'tests.json'
            with open(tests_path, 'w') as fh:
                json.dump(self.ctx['children'], fh)
            resource = self.create_resource(self.descr, tests_path, APP_HOST_BENCHMARK_TESTS)
            self.mark_resource_ready(resource)


__Task__ = BenchmarkAppHostCustomBinary
