# -*- coding: utf-8 -*-

import json
import logging
import os

from sandbox.projects import resource_types
from sandbox.projects.common.dolbilka import stats_parser
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.channel import channel


CURRENT_DIR = os.path.dirname(os.path.realpath(__file__))
TEMPLATE = 'template.j2'

BIGGER_IS_BETTER = True

IS_BIGGER_BETTER = {
    "SELF-SS-AverageResponseTimeMcs": not BIGGER_IS_BETTER,
    "SELF-SS-Failures": not BIGGER_IS_BETTER,
    "SELF-SS-Requests": BIGGER_IS_BETTER,
    "SELF-SS-ResponseSizeHistogram": not BIGGER_IS_BETTER,
    "SELF-SS-ResponseTimesMcsHistogram": not BIGGER_IS_BETTER,
    "SELF-SS-Successes": BIGGER_IS_BETTER,
    "AppHostResponseTime": not BIGGER_IS_BETTER,
    "Errors": not BIGGER_IS_BETTER,
    "ResponseTime": not BIGGER_IS_BETTER,
    "Successes": BIGGER_IS_BETTER,
    'Ok': BIGGER_IS_BETTER,
    'requests': BIGGER_IS_BETTER,
    'requests/sec': BIGGER_IS_BETTER,
    'avg req. time': not BIGGER_IS_BETTER,
    'req time std deviation': not BIGGER_IS_BETTER,
}

COLORS = {
    'very': 'very',
    'good': 'good',
    'none': 'none',
    'missing': 'missing',
    'warn': 'warn',
    'crit': 'crit',
}

CODES = [
    ['none', 'none'],
    ['good', 'warn'],
    ['very', 'crit']
]


class PreviousConfiguration(parameters.ResourceSelector):
    name = 'previous_tests'
    description = 'Previous benchmarks'
    resource_type = resource_types.APP_HOST_BENCHMARK_TESTS


class LastConfiguration(parameters.ResourceSelector):
    name = 'last_tests'
    description = 'Last benchmarks'
    resource_type = resource_types.APP_HOST_BENCHMARK_TESTS


class BenchmarkAppHostDiff2(SandboxTask):
    type = "BENCHMARK_APP_HOST_DIFF2"

    execution_space = 4096
    default_cpu_model = None

    input_parameters = [PreviousConfiguration, LastConfiguration]

    environment = (environments.PipEnvironment('Jinja2', '2.8', use_wheel=True),)
    template = None

    def parse_executor_stat(self, task_id):
        disabled_metrics = ['end time', 'start time', 'data readed']
        executor_stat = channel.sandbox.list_resources(resource_type='EXECUTOR_STAT', task_id=task_id, status="READY")
        if executor_stat is None:
            return 'EXECUTOR_STAT', {'EXECUTOR': {}}
        res_id = executor_stat[0].id
        res_path = self.sync_resource(res_id)
        stats = stats_parser.StatsParser(res_path)
        return 'EXECUTOR_STAT', {'EXECUTOR': {k: v for k, v in stats.vars.iteritems() if k not in disabled_metrics}}

    def parse_benchmark_metrics(self, task_id):
        ret = {}

        logging.debug("BENCHMARK_METRICS: {}".format(task_id))
        resources = channel.sandbox.list_resources(resource_type='BENCHMARK_METRICS', task_id=task_id, status="READY")
        logging.debug("BENCHMARK_METRICS result: {}".format(resources))

        if resources:
            for resource in resources:
                res_id = resource.id
                res_app = resource.attributes.get('app')
                res_path = self.sync_resource(res_id)
                with open(res_path) as fp:
                    res_json = json.load(fp)

                ret[res_app] = res_json

        return 'BENCHMARK_METRICS', ret

    def get_parsed_info(self, test_names, previous, current):
        res = {}
        for k, test in [('previous', previous), ('current', current)]:
            res[k] = {}
            for test_name in test_names:
                res.get(k)[test_name] = {}
                pid = test.get(test_name)
                for name, result in [self.parse_executor_stat(pid), self.parse_benchmark_metrics(pid)]:
                    logging.debug("{} {}".format(name, result))
                    res.get(k).get(test_name)[name] = result
        logging.debug("get_parsed_info {}".format(res))
        return res

    def on_execute(self):
        import jinja2
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(CURRENT_DIR))
        template = env.get_template(TEMPLATE)

        with self.current_action('Downloading dependencies'):
            # Download previous configuration
            path = self.sync_resource(self.ctx.get(PreviousConfiguration.name))
            with open(path, 'r') as fh:
                previous = json.load(fh)
            logging.debug('prev configuration: {}'.format(previous))

            # Download last configuration
            path = self.sync_resource(self.ctx.get(LastConfiguration.name))
            with open(path, 'r') as fh:
                current = json.load(fh)
            logging.debug('current configuration: {}'.format(current))

        added = set(set(current.keys()) - set(previous.keys()))
        removed = set(set(previous.keys()) - set(current.keys()))
        union = set(current.keys()).union(set(previous.keys()))
        logging.warn('tests only in prev configuration: {}'.format(removed))
        logging.warn('tests only in current configuration: {}'.format(added))
        data = self.get_parsed_info(union, previous, current)
        logging.debug("rendering template")
        rendered_template = template.render(diff_list=self.prepare_diff_data(data))
        logging.debug(rendered_template)
        self.ctx['render'] = rendered_template

    def prepare_diff_data(self, data):
        self.ctx['data'] = data
        curr = data.get('current')
        prev = data.get('previous')
        metric_list = []

        for benchmark, benchmark_results in curr.iteritems():
            for metric_type, metric_apps in benchmark_results.iteritems():
                for app, app_results in metric_apps.iteritems():
                    for metric, metric_value in app_results.iteritems():
                        if type(metric_value) == list:
                            for quant, result_value in metric_value:
                                metric_name = metric
                                metric_list.append([benchmark, metric_type, app, metric_name, [quant, result_value]])
                        else:
                            metric_name = metric
                            result_value = metric_value
                            metric_list.append([benchmark, metric_type, app, metric_name, result_value])

        diff_list = []
        for metric in metric_list:
            benchmark, metric_type, app, signal, value = metric
            res = prev.get(benchmark, {}).get(metric_type, {}).get(app, {}).get(signal)
            if app != 'EXECUTOR':
                signal = '_'.join(signal.split('_')[:-1])
            name = signal
            human_name = signal
            curr_value = value
            prev_value = res
            if type(value) == list:
                curr_value = value[1]
            if type(res) == list:
                val = [[q, v] for q, v in res if q == value[0]]
                if val:
                    prev_value = val[0][1]
                else:
                    prev_value = None

                humanq = value[0] * 100
                if value[0] > 0.99:
                    humanq = value[0] * 1000
                if value[0] > 0.999:
                    humanq = value[0] * 10000

                human_name = "{} q{}".format(signal, int(humanq))

            diff, color = self.calculate_diff(name, curr_value, prev_value)
            diff_list.append([benchmark, app, human_name, prev_value, curr_value, diff, color])

        return diff_list

    @staticmethod
    def calculate_diff(name, current, prev):
        if current is None and prev is None:
            no_diff = False
            diff = 0
        elif current == 0 and prev == 0:
            no_diff = False
            diff = 0
        elif current is not None and prev:
            diff = round((float(current) - float(prev)) / float(prev) * 100.0, 2)
            no_diff = False
        else:
            no_diff = True
            diff = 0

        if no_diff:
            color = COLORS.get('missing')

        if diff == 0 and not no_diff:
            color = COLORS.get('none')

        if abs(diff) <= 2.0:
            color_code = 0
        if 2.0 < abs(diff) <= 7.5:
            color_code = 1
        if abs(diff) > 7.5:
            color_code = 2

        if IS_BIGGER_BETTER.get(name):
            if diff < 0:
                color = COLORS.get(CODES[color_code][1])
            elif diff > 0:
                color = COLORS.get(CODES[color_code][0])
        else:
            if diff > 0:
                color = COLORS.get(CODES[color_code][1])
            elif diff < 0:
                color = COLORS.get(CODES[color_code][0])

        return diff, color

    @property
    def footer(self):
        if not self.is_completed():
            return None
        return [{
            "helperName": "",
            "content": self.ctx.get('render')
        }]
