# -*- coding: utf-8 -*-

from sandbox import sdk2

import json
import logging

from sandbox.projects.app_host.BuildAppHostEarlDataBundle import AppHostEarlDataBundle, BuildAppHostEarlDataBundle

from sandbox.common.types.misc import NotExists
from sandbox.common.types.task import Status, ReleaseStatus


class BuildAndReleaseAppHostEarlDataBundle(sdk2.Task):
    """Builds and releases Earl data bundle including graphs and backends for all verticals"""

    class Requirements(sdk2.Task.Requirements):
        disk_space = 4 * 1024

    class Parameters(sdk2.Task.Parameters):

        kill_timeout = 24 * 60 * 60

        check_interval = sdk2.parameters.Integer(
            'Check interval in seconds',
            default=60,
            required=True
        )

        check_count = sdk2.parameters.Integer(
            'Checks count',
            default=60,
            required=True
        )

    def on_execute(self):
        if self.Context.sub_task_id:
            sdk2.Task.server.release(
                task_id=self.Context.sub_task_id,
                type=ReleaseStatus.STABLE,
                subject=self.Parameters.description
            )
            return

        prev_versions = {}

        latest_bundle = sdk2.Resource.find(
            AppHostEarlDataBundle,
            attrs=dict(released=ReleaseStatus.STABLE)
        ).order(-sdk2.Resource.id).first()

        if latest_bundle:
            prev_versions = json.loads(latest_bundle.contents)
            logging.info(
                "Using previous bundle {} with contents {}".format(
                    latest_bundle,
                    latest_bundle.contents
                )
            )

        res_to_reload = []

        for t, handler in BuildAppHostEarlDataBundle.resource_handlers.iteritems():
            for release_type in handler.release_types():
                current = sdk2.Resource.find(
                    handler.resource_type(),
                    attrs=dict(released=release_type)
                ).order(-sdk2.Resource.id).first()

                if current:
                    prev_version = prev_versions.get(t, {}).get(release_type)

                    if not latest_bundle or current.id > (prev_version or 0):
                        res_to_reload.append(current.id)
                        logging.info(
                            "Newer {} resource found: {}, was {}".format(
                                release_type, current, prev_version
                            )
                        )

        if latest_bundle and not res_to_reload:
            if self.Context.runs is NotExists:
                self.Context.runs = 0

            if self.Context.runs >= self.Parameters.check_count:
                return

            logging.info("Nothing to update, going to wait {} seconds".format(self.Parameters.check_interval))

            self.Context.runs += 1

            raise sdk2.WaitTime(self.Parameters.check_interval)
        else:
            descr = "{} #{}: {} with {}".format(self.Parameters.description, self.id, latest_bundle, res_to_reload)
            sub_task = BuildAppHostEarlDataBundle(
                self,
                description=descr,
                notifications=self.Parameters.notifications,
                previous_bundle=latest_bundle,
                new_releases=res_to_reload,
                create_sub_task=False
            ).enqueue()

            self.Context.sub_task_id = sub_task.id

            raise sdk2.WaitTask([sub_task], Status.Group.FINISH | Status.Group.BREAK)
