import logging

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.sandboxapi import TASK_FINISHED

from sandbox.projects.app_host.BuildAppHostSrcSetup import BuildAppHostSrcSetup
from sandbox.projects.app_host.BuildAppHostSrcSetupConfigBundle import BuildAppHostSrcSetupConfigBundle, BackendsConfigTask

from sandbox.projects.common.app_host.options import get_verticals
from sandbox.projects.common.app_host.regular_build import save_run_info, has_changes_since_prev_run

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask


class OptionalBackendsConfigTask(BackendsConfigTask):
    required = False


def _build_input_params():
    params = list(BuildAppHostSrcSetup.input_parameters)
    for p in BuildAppHostSrcSetupConfigBundle.input_parameters:
        if p is BackendsConfigTask:
            params.append(OptionalBackendsConfigTask)
        elif p not in params:
            params.append(p)
    return params


class BuildAndReleaseAppHostSrcSetupBundle(ArcadiaTask):

    """Runs and releases BUILD_APP_HOST_SRC_SETUP and BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE tasks"""

    type = 'BUILD_AND_RELEASE_APP_HOST_SRC_SETUP_BUNDLE'

    execution_space = 100000

    input_parameters = _build_input_params()

    def _init_auto_params(self):
        if self.ctx[BackendsConfigTask.name]:
            return

        tasks = channel.sandbox.list_tasks(
            task_type=BackendsConfigTask.task_type,
            status=TASK_FINISHED,
            limit=100
        )

        logging.info("[_init_auto_params] tasks: {}".format(len(tasks)))

        tasks = sorted(tasks, key=lambda t: -t.id)

        verticals = set(get_verticals(self.ctx))
        descr = "task '%s' for verticals '%s'" % (BackendsConfigTask.task_type, " ".join(verticals))

        for t in tasks:
            if set(get_verticals(t.ctx)) == verticals:
                # Do not search for backends config task
                # self.ctx[BackendsConfigTask.name] = t.id

                logging.info("Found %s, id %d" % (descr, t.id))
                return
            else:
                logging.info("[_init_auto_params] skip task {} | {}".format(t.id, get_verticals(t.ctx)))

        raise SandboxTaskFailureError("Failed to find suitable " + descr)

    def _get_subtasks(self):
        key = 'src_setup_bundle_tasks'
        if key not in self.ctx:
            self.ctx[key] = list()
        return self.ctx[key]

    def _run_subtask(self, subtask_type):
        subtask = self.create_subtask(
            task_type=subtask_type,
            description=self.descr,
            input_parameters=self.ctx,
            execution_space=self.execution_space,
            priority=self.priority,
            inherit_notifications=True
        )
        self._get_subtasks().append(subtask.id)
        return

    def do_execute(self):

        subtasks = self._get_subtasks()
        if not subtasks:
            save_run_info(self.ctx)
            has_changes = has_changes_since_prev_run(self.ctx, BuildAndReleaseAppHostSrcSetupBundle.type, '/web/src_setup')

            if has_changes:
                self._run_subtask('BUILD_APP_HOST_SRC_SETUP')
                self._run_subtask('BUILD_APP_HOST_SRC_SETUP_CONFIG_BUNDLE')
                self.wait_tasks(
                    self._get_subtasks(),
                    self.Status.Group.FINISH | self.Status.Group.BREAK,
                    True,
                    None
                )
        else:
            for task_id in subtasks:
                self.create_release(task_id, status='unstable', subject="{}".format(self.descr))


__Task__ = BuildAndReleaseAppHostSrcSetupBundle
